package main

import (
	"context"
	"os"
	"time"

	"github.com/aws/aws-lambda-go/lambda"

	"code.justin.tv/businessviewcount/aperture/config"
	"code.justin.tv/businessviewcount/aperture/internal/clients/secrets"
	"code.justin.tv/businessviewcount/aperture/internal/clients/stats"
	"code.justin.tv/foundation/twitchclient"

	pb "code.justin.tv/businessviewcount/aperture/rpc/aperture"
	log "github.com/sirupsen/logrus"
)

// spadeLoggerLambda implements the lambda to call the spade logger and write channel concurrents to spade
type spadeLoggerLambda struct {
	apertureClient pb.Aperture
	statsdClient   stats.StatSender
}

// Handler update channel concurrents in spade
func (l *spadeLoggerLambda) Handler(ctx context.Context) error {
	start := time.Now()
	req := &pb.UpdateSpadeViewcountReq{}

	_, err := l.apertureClient.UpdateSpadeViewcount(ctx, req)

	elapsed := time.Since(start)
	l.statsdClient.ExecutionTime("spade_logger.execution_time", elapsed)

	return err
}

func main() {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("spade_logger: no environment found in env")
		return
	}

	conf := &config.Config{
		Environment: env,
	}

	err := conf.Load()
	if err != nil {
		log.Fatal("spade_logger: could not load config: ", err)
		return
	}

	secretManager, err := secrets.NewManager()
	if err != nil {
		log.Fatal("spade_logger: could not create secrets manager: ", err)
		return
	}

	statsdClient, err := stats.NewClient(conf.StatsdHost.Get(), env)
	if err != nil {
		log.Fatal("spade_logger: could not create statsd client: ", err)
		return
	}

	config.SetupRollbarLogging(secretManager,
		conf.RollbarTokenSecretName.Get(),
		conf.RollbarTokenSecretKey.Get(),
		env)

	apertureHost := conf.ApertureHost.Get()
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host: apertureHost,
	}

	apertureClient := pb.NewApertureProtobufClient(apertureHost, twitchclient.NewHTTPClient(clientConf))
	spadeLogger := spadeLoggerLambda{
		apertureClient,
		statsdClient,
	}

	lambda.Start(spadeLogger.Handler)
}
