# Elastic Beanstalk App
resource "aws_iam_role" "app" {
  name = "${var.name}-${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        ${var.allow_assume_role_arn != "" ? "\"AWS\": \"${var.allow_assume_role_arn}\"," : ""}
        "Service": "ec2.amazonaws.com"
      },
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_instance_profile" "app" {
  name = aws_iam_role.app.name
  role = aws_iam_role.app.name
}

resource "aws_iam_role_policy" "policy" {
  name       = "${var.name}_${var.environment}_policy"
  role       = aws_iam_role.app.id
  depends_on = ["aws_iam_role.app"]

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": [
        "arn:aws:logs:us-west-2:*:*",
        "arn:aws:logs:*:*:log-group:/aws/elasticbeanstalk*"
      ]
    },
    {
      "Effect": "Allow",
      "Action": "s3:GetObject",
      "Resource": [
          "arn:aws:s3:::minixperiment-prod/*"
      ]
    }
  ]
}
EOF
}

# IAM user that Jenkins uses for builds and deploys.
resource "aws_iam_user" "jenkins_user" {
  name = "${var.name}-deploy-${var.environment}"
}

# Attach the AWSElasticBeanstalkFullAccess policy to the Jenkins user so that
# it can deploy to Elastic Beanstalk.
resource "aws_iam_user_policy_attachment" "jenkins_user_beanstalk_attachment" {
  user       = aws_iam_user.jenkins_user.name
  policy_arn = "arn:aws:iam::aws:policy/AWSElasticBeanstalkFullAccess"
}

# Attach the default Read/Write policy to the beanstalk role so we can
# get secrets from AWS secret manager
resource "aws_iam_role_policy_attachment" "beanstalk_role_secrets_attachment" {
  role       = aws_iam_role.app.name
  policy_arn = "arn:aws:iam::aws:policy/SecretsManagerReadWrite"
}

# Attach the default Read-only policy to the beanstalk role so we can
# get configuration parameters from AWS systems manager
resource "aws_iam_role_policy_attachment" "beanstalk_role_ssm_attachment" {
  role       = aws_iam_role.app.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMReadOnlyAccess"
}
