resource "aws_elasticache_subnet_group" "aperture_cache" {
  # Subnet group names can only use alphanumeric and hyphens
  name        = "cb-${var.name}-${var.environment}"
  description = "Subnet group for memcached cluster used in Aperture"
  subnet_ids  = var.private_subnets
}

resource "aws_elasticache_cluster" "aperture_cache" {
  # Cluster IDs must be less than 20 characters long and only contain lowercase characters or hyphens
  cluster_id           = "cb-${var.name}-${var.environment}"
  engine               = "memcached"
  port                 = 11211
  node_type            = var.elasticache_node_type
  num_cache_nodes      = var.elasticache_node_count
  apply_immediately    = true
  parameter_group_name = aws_elasticache_parameter_group.aperture_memcached.name
  security_group_ids   = [var.security_group]
  subnet_group_name    = aws_elasticache_subnet_group.aperture_cache.name
  az_mode              = "cross-az"

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team
  }
}

resource "aws_elasticache_replication_group" "aperture_redis_cluster" {
  replication_group_id          = "cb-redis-${var.name}-${var.environment}"
  replication_group_description = "redis cluster for ${var.name} ${var.environment}"
  engine                        = "redis"
  port                          = 6379
  node_type                     = var.elasticache_redis_node_type
  security_group_ids            = [var.security_group]
  subnet_group_name             = aws_elasticache_subnet_group.aperture_cache.name
  parameter_group_name          = aws_elasticache_parameter_group.aperture_redis_cluster.name
  engine_version                = "5.0.6"

  automatic_failover_enabled = true
  auto_minor_version_upgrade = true
  apply_immediately          = true
  maintenance_window         = "mon:06:00-mon:07:00"

  snapshot_retention_limit = 1
  snapshot_window          = "10:00-11:00"

  cluster_mode {
    num_node_groups         = 1
    replicas_per_node_group = var.elasticache_redis_num_nodes
  }

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team
  }

  lifecycle {
    prevent_destroy = true
  }
}
