resource "aws_elasticache_parameter_group" "aperture_memcached" {
  name        = "aperture-memcached-${var.environment}"
  family      = "memcached1.5"
  description = "Extends the default memcached parameter group"

  # Increase max_item_size to 9,437,184 bytes from the default 1,048,576 bytes.
  # This number was chosen because in some limited testing, cache item sizes for
  # all viewcount data were ~2,330,000 bytes for ~50k live channels.
  # The chosen max size is ~4x of that observered size, with some additional bytes
  # because the size has to be an exact multiple of the slab_chunk_max parameter.
  # The maximum allowable size in memcached is 134,217,728 bytes, which we are well under.
  parameter {
    name  = "max_item_size"
    value = "9437184"
  }
}

# Parameter group copied from https://git.xarth.tv/chat/terraform/blob/master/infrastructure/twitch-chat-aws/elasticache/twitch_redis_cluster.tf
resource "aws_elasticache_parameter_group" "aperture_redis_cluster" {
  name        = "${var.name}-${var.environment}-redis-cluster"
  family      = "redis5.0"
  description = "Uses parameter group as defined by the chat team"

  // Enable cluster mode.
  parameter {
    name  = "cluster-enabled"
    value = "yes"
  }

  // Sets the eviction policy for the cluster. Default value: volatile-lru.
  //
  // noeviction: Returns errors instead of enabling LRU.
  // volatile-lru: Enables LRU on keys with TTLs only.
  //   "The policies volatile-lru, volatile-random and volatile-ttl behave like
  //    noeviction if there are no keys to evict matching the prerequisites."
  // allkeys-lru: Enables LRU across all keys (recommended in most cases).
  //   "Use the allkeys-lru policy when you expect a power-law distribution in
  //   the popularity of your requests, that is, you expect that a subset of
  //   elements will be accessed far more often than the rest."
  //
  // https://redis.io/topics/lru-cache#eviction-policies
  parameter {
    name  = "maxmemory-policy"
    value = "allkeys-lru"
  }

  // Maximum number of memory samples. Default value: 3.
  //
  // For least-recently-used (LRU) and time-to-live (TTL) calculations, this
  // parameter represents the sample size of keys to check. By default, Redis
  // chooses 3 keys and uses the one that was used least recently.
  // "Using a sample size of 10 in Redis 3.0, the approximation is very close
  // to the theoretical performance of [a non-simulated LRU]."
  // "If your data access pattern closely resembles the power law, most of the
  // accesses will be in the set of keys that the LRU approximation will be
  // able to handle well."
  // "You can raise the sample size to 10 at the cost of some additional CPU
  // usage in order to closely approximate true LRU."
  //
  // https://redis.io/topics/lru-cache#approximated-lru-algorithm
  // https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/ParameterGroups.Redis.html#ParameterGroups.Redis.2-6-13
  parameter {
    name  = "maxmemory-samples"
    value = "10"
  }

  // The execution time, in microseconds, to exceed in order for the command
  // to be logged in the Redis slow log. A negative number disables the log,
  // while a zero value logs every command. Default value: 10000us.
  //
  // https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/ParameterGroups.Redis.html#ParameterGroups.Redis.2-6-13
  parameter {
    name  = "slowlog-log-slower-than"
    value = "10000"
  }

  // If nonzero, send ACKs every given number of seconds. Default value: 300.
  //
  // https://redis.io/topics/clients#tcp-keepalive
  parameter {
    name  = "tcp-keepalive"
    value = "300"
  }

  // The percentage of memory reserved for non-cache memory usage. This helps
  // prevent Redis nodes from swapping when memory is allocated unexpectedly,
  // such as during a replication or recovery event.
  //
  // https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/ParameterGroups.Redis.html#ParameterGroups.Redis.3-2-4
  // https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/redis-memory-management.html
  parameter {
    name  = "reserved-memory-percent"
    value = "25"
  }
}
