# Due to a terraform bug (https://github.com/terraform-providers/terraform-provider-aws/issues/214), if we try to
# create alarms at the CacheNodeId level, it will fail on `terraform apply`.

# More info @ https://git.xarth.tv/chat/terraform/tree/master/modules/elasticache_redis/alarms

locals {
  node_type_to_maxmemory = {
    "cache.c1.xlarge"   = "6501171200"
    "cache.m1.large"    = "7025459200"
    "cache.m1.medium"   = "3093299200"
    "cache.m1.small"    = "943718400"
    "cache.m1.xlarge"   = "14889779200"
    "cache.m2.2xlarge"  = "35022438400"
    "cache.m2.4xlarge"  = "70883737600"
    "cache.m2.xlarge"   = "17091788800"
    "cache.m3.2xlarge"  = "29989273600"
    "cache.m3.large"    = "6501171200"
    "cache.m3.medium"   = "2988441600"
    "cache.m3.xlarge"   = "14260633600"
    "cache.m4.10xlarge" = "166047614239"
    "cache.m4.2xlarge"  = "31889126359"
    "cache.m4.4xlarge"  = "65257290629"
    "cache.m4.large"    = "6892593152"
    "cache.m4.xlarge"   = "15328501760"
    "cache.m5.12xlarge" = "168715971994"
    "cache.m5.24xlarge" = "337500562842"
    "cache.m5.2xlarge"  = "27966669210"
    "cache.m5.4xlarge"  = "56116178125"
    "cache.m5.large"    = "6854542746"
    "cache.m5.xlarge"   = "13891921715"
    "cache.r3.2xlarge"  = "62495129600"
    "cache.r3.4xlarge"  = "126458265600"
    "cache.r3.8xlarge"  = "254384537600"
    "cache.r3.large"    = "14470348800"
    "cache.r3.xlarge"   = "30513561600"
    "cache.r4.16xlarge" = "437021573120"
    "cache.r4.2xlarge"  = "54197537997"
    "cache.r4.4xlarge"  = "108858546586"
    "cache.r4.8xlarge"  = "218255432090"
    "cache.r4.large"    = "13201781556"
    "cache.r4.xlarge"   = "26898228839"
    "cache.r5.12xlarge" = "341206346547"
    "cache.r5.24xlarge" = "682485973811"
    "cache.r5.2xlarge"  = "56711183565"
    "cache.r5.4xlarge"  = "113609865216"
    "cache.r5.large"    = "14037181030"
    "cache.r5.xlarge"   = "28261849702"
    "cache.t1.micro"    = "142606336"
    "cache.t2.medium"   = "3461349376"
    "cache.t2.micro"    = "581959680"
    "cache.t2.small"    = "1665138688"
  }

  node_type_to_vcpu_count = {
    // vCPU counts can be found on https://aws.amazon.com/elasticache/pricing/ and https://aws.amazon.com/elasticache/previous-generation/
    "cache.c1.xlarge"   = 8
    "cache.m1.small"    = 1
    "cache.m1.medium"   = 1
    "cache.m1.large"    = 2
    "cache.m1.xlarge"   = 4
    "cache.m2.xlarge"   = 2
    "cache.m2.2xlarge"  = 4
    "cache.m2.4xlarge"  = 8
    "cache.m3.medium"   = 1
    "cache.m3.large"    = 2
    "cache.m3.xlarge"   = 4
    "cache.m3.2xlarge"  = 8
    "cache.m4.large"    = 2
    "cache.m4.xlarge"   = 4
    "cache.m4.2xlarge"  = 8
    "cache.m4.4xlarge"  = 16
    "cache.m4.10xlarge" = 40
    "cache.m5.large"    = 2
    "cache.m5.xlarge"   = 4
    "cache.m5.2xlarge"  = 8
    "cache.m5.4xlarge"  = 16
    "cache.m5.12xlarge" = 48
    "cache.m5.24xlarge" = 96
    "cache.r3.large"    = 2
    "cache.r3.xlarge"   = 4
    "cache.r3.2xlarge"  = 8
    "cache.r3.4xlarge"  = 16
    "cache.r3.8xlarge"  = 32
    "cache.r4.large"    = 2
    "cache.r4.xlarge"   = 4
    "cache.r4.2xlarge"  = 8
    "cache.r4.4xlarge"  = 16
    "cache.r4.8xlarge"  = 32
    "cache.r4.16xlarge" = 64
    "cache.r5.large"    = 2
    "cache.r5.xlarge"   = 4
    "cache.r5.2xlarge"  = 8
    "cache.r5.4xlarge"  = 16
    "cache.r5.12xlarge" = 48
    "cache.r5.24xlarge" = 96
    "cache.t1.micro"    = 1
    "cache.t2.micro"    = 1
    "cache.t2.small"    = 1
    "cache.t2.medium"   = 2
  }

  # For determining if EngineCPUUtilization or CPUUtilization should be monitored
  greater_than_two_vcpus = local.node_type_to_vcpu_count[var.node_type] > 2 ? 1 : 0
}

# Engine CPU only if there are greater than 2 vCPUs per AWS recommendations
resource "aws_cloudwatch_metric_alarm" "engine_cpu" {
  count = local.greater_than_two_vcpus == 1 ? var.num_cache_clusters : 0

  alarm_name          = "${format("%s-%03d", var.cluster_id_prefix, count.index + 1)}-elasticache-cluster-engine-cpu"
  alarm_description   = "Alerts when EngineCPUUtilization for ClusterId ${format("%s-%03d", var.cluster_id_prefix, count.index + 1)} exceeds ${var.engine_cpu_utilization_threshold}% for ${var.engine_cpu_utilization_period * var.engine_cpu_utilization_evaluation_periods} seconds"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.engine_cpu_utilization_evaluation_periods
  metric_name         = "EngineCPUUtilization"
  namespace           = "AWS/ElastiCache"
  period              = var.engine_cpu_utilization_period
  statistic           = "Average"
  threshold           = var.engine_cpu_utilization_threshold

  dimensions = {
    CacheClusterId = format("%s-%03d", var.cluster_id_prefix, count.index + 1)
  }

  alarm_actions   = var.action_arns
  ok_actions      = var.action_arns
  actions_enabled = true
}

# CPU only if there are less than or equal to 2 vCPUs per AWS recommendations
locals {
  cpu_utilization_threshold = var.engine_cpu_utilization_threshold / local.node_type_to_vcpu_count[var.node_type]
}

resource "aws_cloudwatch_metric_alarm" "cpu" {
  count = local.greater_than_two_vcpus == 0 ? var.num_cache_clusters : 0

  alarm_name          = "${format("%s-%03d", var.cluster_id_prefix, count.index + 1)}-elasticache-cluster-cpu"
  alarm_description   = "Alerts when CPUUtilization for ClusterId ${format("%s-%03d", var.cluster_id_prefix, count.index + 1)} exceeds ${local.cpu_utilization_threshold}% (${var.engine_cpu_utilization_threshold}% / ${local.node_type_to_vcpu_count[var.node_type]} vCPU) for ${var.engine_cpu_utilization_period * var.engine_cpu_utilization_evaluation_periods} seconds"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.engine_cpu_utilization_evaluation_periods
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ElastiCache"
  period              = var.engine_cpu_utilization_period
  statistic           = "Average"
  threshold           = local.cpu_utilization_threshold

  dimensions = {
    CacheClusterId = format("%s-%03d", var.cluster_id_prefix, count.index + 1)
  }

  alarm_actions   = var.action_arns
  ok_actions      = var.action_arns
  actions_enabled = true
}

# Evictions
resource "aws_cloudwatch_metric_alarm" "evictions" {
  count = var.num_cache_clusters

  alarm_name          = "${format("%s-%03d", var.cluster_id_prefix, count.index + 1)}-elasticache-cluster-evictions"
  alarm_description   = "Alerts when Evictions for ClusterId ${format("%s-%03d", var.cluster_id_prefix, count.index + 1)} exceeds ${var.evictions_threshold} for ${var.evictions_period * var.evictions_evaluation_periods} seconds"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.evictions_evaluation_periods
  metric_name         = "Evictions"
  namespace           = "AWS/ElastiCache"
  period              = var.evictions_period
  statistic           = "Average"
  threshold           = var.evictions_threshold

  dimensions = {
    CacheClusterId = format("%s-%03d", var.cluster_id_prefix, count.index + 1)
  }

  alarm_actions   = var.action_arns
  ok_actions      = var.action_arns
  actions_enabled = true
}

# CurrConnections
resource "aws_cloudwatch_metric_alarm" "curr_connections" {
  count = var.num_cache_clusters

  alarm_name          = "${format("%s-%03d", var.cluster_id_prefix, count.index + 1)}-elasticache-cluster-curr_connections"
  alarm_description   = "Alerts when CurrConnections for ClusterId ${format("%s-%03d", var.cluster_id_prefix, count.index + 1)} exceeds ${var.curr_connections_threshold} for ${var.curr_connections_datapoints_to_alarm > 0 ? var.curr_connections_datapoints_to_alarm : var.curr_connections_evaluation_periods} of ${var.curr_connections_evaluation_periods} ${var.curr_connections_period} second periods."
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.curr_connections_evaluation_periods
  metric_name         = "CurrConnections"
  namespace           = "AWS/ElastiCache"
  datapoints_to_alarm = var.curr_connections_datapoints_to_alarm > 0 ? var.curr_connections_datapoints_to_alarm : var.curr_connections_evaluation_periods
  period              = var.curr_connections_period
  statistic           = "Average"
  threshold           = var.curr_connections_threshold

  dimensions = {
    CacheClusterId = format("%s-%03d", var.cluster_id_prefix, count.index + 1)
  }

  alarm_actions   = var.action_arns
  ok_actions      = var.action_arns
  actions_enabled = true
}

# NewConnections
resource "aws_cloudwatch_metric_alarm" "new_connections" {
  count = var.num_cache_clusters

  alarm_name          = "${format("%s-%03d", var.cluster_id_prefix, count.index + 1)}-elasticache-cluster-new_connections"
  alarm_description   = "Alerts when NewConnections for ClusterId ${format("%s-%03d", var.cluster_id_prefix, count.index + 1)} exceeds ${var.new_connections_threshold} for ${var.new_connections_period * var.new_connections_evaluation_periods} seconds"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.new_connections_evaluation_periods
  metric_name         = "NewConnections"
  namespace           = "AWS/ElastiCache"
  period              = var.new_connections_period
  statistic           = "Average"
  threshold           = var.new_connections_threshold

  dimensions = {
    CacheClusterId = format("%s-%03d", var.cluster_id_prefix, count.index + 1)
  }

  alarm_actions   = var.action_arns
  ok_actions      = var.action_arns
  actions_enabled = true
}

locals {
  maxmemory                                 = local.node_type_to_maxmemory[var.node_type]
  bytes_used_for_cache_perc_threshold_ratio = var.bytes_used_for_cache_perc_threshold / 100
  bytes_used_for_cache_threshold            = ceil(ceil(local.maxmemory - local.maxmemory * var.reserved_memory_percent / 100) * local.bytes_used_for_cache_perc_threshold_ratio)
}

# BytesUsedForCache
resource "aws_cloudwatch_metric_alarm" "bytes_used_for_cache" {
  count = var.num_cache_clusters

  alarm_name          = "${format("%s-%03d", var.cluster_id_prefix, count.index + 1)}-elasticache-cluster-bytes_used_for_cache"
  alarm_description   = "Alerts when BytesUsedForCache for ClusterId ${format("%s-%03d", var.cluster_id_prefix, count.index + 1)} exceeds ${local.bytes_used_for_cache_threshold} bytes (${var.bytes_used_for_cache_perc_threshold}% usage) for ${var.bytes_used_for_cache_period * var.bytes_used_for_cache_evaluation_periods} seconds"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.bytes_used_for_cache_evaluation_periods
  metric_name         = "BytesUsedForCache"
  namespace           = "AWS/ElastiCache"
  period              = var.bytes_used_for_cache_period
  statistic           = "Average"
  threshold           = local.bytes_used_for_cache_threshold

  dimensions = {
    CacheClusterId = format("%s-%03d", var.cluster_id_prefix, count.index + 1)
  }

  alarm_actions   = var.action_arns
  ok_actions      = var.action_arns
  actions_enabled = true
}

# SwapUsage
resource "aws_cloudwatch_metric_alarm" "swap_usage" {
  count = var.num_cache_clusters

  alarm_name          = "${format("%s-%03d", var.cluster_id_prefix, count.index + 1)}-elasticache-cluster-swap_usage"
  alarm_description   = "Alerts when SwapUsage for ClusterId ${format("%s-%03d", var.cluster_id_prefix, count.index + 1)} exceeds ${var.swap_usage_threshold}MB for ${var.swap_usage_evaluation_periods} ${var.swap_usage_evaluation_periods * var.swap_usage_period} seconds."
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.swap_usage_evaluation_periods
  metric_name         = "SwapUsage"
  namespace           = "AWS/ElastiCache"
  period              = var.swap_usage_period
  statistic           = "Average"

  # Convert to megabytes
  threshold = var.swap_usage_threshold * 1024 * 1024

  dimensions = {
    CacheClusterId = format("%s-%03d", var.cluster_id_prefix, count.index + 1)
  }

  alarm_actions   = var.action_arns
  ok_actions      = var.action_arns
  actions_enabled = true
}
