resource "aws_kinesis_analytics_application" "minute_watched_ratio_app" {
  name = "cb-${var.name}-${var.environment}-minute-watched-ratio-application"

  inputs {
    name_prefix = "MINUTE_WATCHED_STREAM"

    parallelism {
      count = 16
    }

    kinesis_stream {
      resource_arn = aws_kinesis_stream.minute_watched_stream.arn
      role_arn     = aws_iam_role.minute_watched_ratio_app_role.arn
    }

    schema {
      record_columns {
        name     = "UUID"
        sql_type = "VARCHAR(64)"
        mapping  = "$.UUID"
      }

      record_columns {
        name     = "Version"
        sql_type = "INTEGER"
        mapping  = "$.Version"
      }

      record_columns {
        name     = "channel_id"
        sql_type = "VARCHAR(32)"
        mapping  = "$.Data.channel_id"
      }

      record_columns {
        name     = "player"
        sql_type = "VARCHAR(32)"
        mapping  = "$.Data.player"
      }

      record_columns {
        name     = "COL_time"
        sql_type = "TIMESTAMP"
        mapping  = "$.Data.time"
      }

      record_columns {
        name     = "CreatedAt"
        sql_type = "TIMESTAMP"
        mapping  = "$.CreatedAt"
      }

      record_encoding = "UTF-8"

      record_format {
        mapping_parameters {
          json {
            record_row_path = "$"
          }
        }
      }
    }
  }

  outputs {
    name = var.output_stream_name

    lambda {
      resource_arn = module.output_mw_ratio_to_elasticache_lambda.lambda_function_arn
      role_arn     = aws_iam_role.minute_watched_ratio_app_role.arn
    }

    schema {
      record_format_type = "JSON"
    }
  }

  outputs {
    name = "error_stream"

    lambda {
      resource_arn = module.error_stream_logger_lambda.lambda_function_arn
      role_arn     = aws_iam_role.minute_watched_ratio_app_role.arn
    }

    schema {
      record_format_type = "JSON"
    }
  }

  code = file("../../kinesis_analytics/mw_ratio.sql")
}

resource "aws_iam_role" "minute_watched_ratio_app_role" {
  name = "cb-${var.name}-${var.environment}-minute-watched-ratio-app-role"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "kinesisanalytics.amazonaws.com"
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "minute_watched_ratio_app_kinesis_policy" {
  name = "${aws_iam_role.minute_watched_ratio_app_role.name}-kinesis-policy"
  role = aws_iam_role.minute_watched_ratio_app_role.id

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "ReadInputKinesis",
            "Effect": "Allow",
            "Action": [
                "kinesis:DescribeStream",
                "kinesis:GetShardIterator",
                "kinesis:GetRecords",
                "kinesis:ListShards"
            ],
            "Resource": [
                "${aws_kinesis_stream.minute_watched_stream.arn}"
            ]
        }
    ]
}
EOF
}
