resource "aws_kinesis_stream" "minute_watched_stream" {
  name        = var.stream_name
  shard_count = var.shard_count

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team
  }
}

locals {
  alarm_sns_topics = split(",", var.pagerduty_sns_topic != "" ? join(",", list(var.scieng_sns_topic, var.pagerduty_sns_topic)) : var.scieng_sns_topic)
}

resource "aws_cloudwatch_metric_alarm" "stream_too_few_bytes" {
  count = var.alarm_on_low_number_of_events

  alarm_name          = "${aws_kinesis_stream.minute_watched_stream.name}-too-few-bytes"
  alarm_description   = "Fewer than 1% of max bytes sent to ${aws_kinesis_stream.minute_watched_stream.name}"
  comparison_operator = "LessThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingBytes"
  namespace           = "AWS/Kinesis"
  period              = "60"
  statistic           = "Sum"
  threshold           = 0.01 * var.shard_count * 1000000 * 60
  treat_missing_data  = "breaching"

  dimensions = {
    StreamName = aws_kinesis_stream.minute_watched_stream.name
  }

  alarm_actions             = local.alarm_sns_topics
  ok_actions                = local.alarm_sns_topics
  insufficient_data_actions = local.alarm_sns_topics
}

resource "aws_cloudwatch_metric_alarm" "stream_too_many_bytes" {
  alarm_name          = "${aws_kinesis_stream.minute_watched_stream.name}-too-many-bytes"
  alarm_description   = "Too many bytes sent to ${aws_kinesis_stream.minute_watched_stream.name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingBytes"
  namespace           = "AWS/Kinesis"
  period              = "60"
  statistic           = "Sum"
  threshold           = 0.85 * var.shard_count * 1000000 * 60
  treat_missing_data  = "notBreaching"

  dimensions = {
    StreamName = aws_kinesis_stream.minute_watched_stream.name
  }

  alarm_actions             = local.alarm_sns_topics
  ok_actions                = local.alarm_sns_topics
  insufficient_data_actions = local.alarm_sns_topics
}

resource "aws_cloudwatch_metric_alarm" "stream_too_many_records" {
  alarm_name          = "${aws_kinesis_stream.minute_watched_stream.name}-too-many-records"
  alarm_description   = "Too many records sent to ${aws_kinesis_stream.minute_watched_stream.name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingRecords"
  namespace           = "AWS/Kinesis"
  period              = "60"
  statistic           = "Sum"
  threshold           = 0.85 * var.shard_count * 1000 * 60
  treat_missing_data  = "notBreaching"

  dimensions = {
    StreamName = aws_kinesis_stream.minute_watched_stream.name
  }

  alarm_actions             = local.alarm_sns_topics
  ok_actions                = local.alarm_sns_topics
  insufficient_data_actions = local.alarm_sns_topics
}

resource "aws_iam_role" "spade_putter" {
  name = "${aws_kinesis_stream.minute_watched_stream.name}-putter"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "AWS": [
          "arn:aws:iam::673385534282:root"
        ]
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "spade_putter_policy" {
  name = "${aws_iam_role.spade_putter.name}-policy"
  role = aws_iam_role.spade_putter.id

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "kinesis:PutRecords"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_kinesis_stream.minute_watched_stream.arn}"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role" "worker" {
  name = "${aws_kinesis_stream.minute_watched_stream.name}-worker"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "worker_policy" {
  name = "${aws_iam_role.worker.name}-policy"
  role = aws_iam_role.worker.id

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "kinesis:DescribeStream",
        "kinesis:GetShardIterator",
        "kinesis:GetRecords"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_kinesis_stream.minute_watched_stream.arn}"
      ]
    }
  ]
}
EOF
}
