resource "aws_iam_role" "spade_logger_lambda_role" {
  name = "cb-${var.name}-${var.environment}-${var.lambda_name}-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Effect": "Allow",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_cloudwatch_log_group" "spade_logger_lambda_log_group" {
  name = "/aws/lambda/${var.name}-${var.environment}-${var.lambda_name}"
}

resource "aws_iam_role_policy_attachment" "spade_logger_lambda_policy_attach" {
  role       = aws_iam_role.spade_logger_lambda_role.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy_attachment" "spade_logger_lambda_ssm_policy_attach" {
  role       = aws_iam_role.spade_logger_lambda_role.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMReadOnlyAccess"
}

resource "aws_iam_role_policy" "spade_logger_lambda_basic_execution" {
  name = "cb-${var.name}-${var.environment}-spade-logger-lambda-basic-execution"
  role = aws_iam_role.spade_logger_lambda_role.id

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "logs:CreateLogStream",
                "logs:PutLogEvents"
            ],
            "Resource": [
                "${aws_cloudwatch_log_group.spade_logger_lambda_log_group.arn}"
            ]
        }
    ]
}
EOF
}

resource "aws_lambda_function" "spade_logger_lambda" {
  function_name = "cb-${var.name}-${var.environment}-${var.lambda_name}"
  filename      = "../../build/lambda/spade_logger/main.zip"
  role          = aws_iam_role.spade_logger_lambda_role.arn
  handler       = "main"
  runtime       = "go1.x"
  description   = "${var.team} team's lambda function to log channel concurrents to spade. Used by ${var.name} service for ${var.environment}"
  timeout       = 180

  vpc_config {
    subnet_ids         = var.private_subnets
    security_group_ids = [var.security_group]
  }

  environment {
    variables = {
      ENVIRONMENT = var.environment
    }
  }

  tags = {
    Owner   = var.owner
    Project = var.project
    Team    = var.team
  }

  lifecycle {
    ignore_changes = ["filename", "source_code_hash", "last_modified"]
  }
}

resource "aws_cloudwatch_event_rule" "spade_logger_event_rule" {
  name                = "cb-${var.name}-${var.environment}-${var.lambda_name}-event"
  description         = "Calls the spade logger lambda every minute"
  schedule_expression = "rate(1 minute)"
}

resource "aws_cloudwatch_event_target" "spade_logger_event_target" {
  rule      = aws_cloudwatch_event_rule.spade_logger_event_rule.name
  target_id = aws_lambda_function.spade_logger_lambda.id
  arn       = aws_lambda_function.spade_logger_lambda.arn
}

resource "aws_lambda_permission" "spade_logger_cloudwatch_permission" {
  statement_id  = "AllowExecutionFromCloudWatch"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.spade_logger_lambda.function_name
  principal     = "events.amazonaws.com"
  source_arn    = aws_cloudwatch_event_rule.spade_logger_event_rule.arn
}
