package usher

import "github.com/golang/protobuf/ptypes/wrappers"
import "github.com/golang/protobuf/ptypes/struct"

// This package contains convenience functions for the various well known
// wrapper types provided in the protobuf specification. Types from:
//
//   google/protobuf/struct.proto
//   google/protobuf/wrappers.proto
//
// are supported for now. These functions help by making things like this:
//
//   val := &wrappers.DoubleValue{Value:1.0}
//   val := &structpb.Value{Kind:&structpb.Value_StringValue{StringValue:"foo"}}
//
// look like
//
//   val := usher.Double(1.0)
//   val := usher.StringValue("foo")
//
// It helps readability quite a bit. Only primitive types are supported, more
// complex use cases need to import the protobuf types directly.

// NumberValue returns a google.protobuf.Value with its value set to val.
func NumberValue(val float64) *structpb.Value {
	return &structpb.Value{
		Kind: &structpb.Value_NumberValue{
			NumberValue: val,
		},
	}
}

// StringValue returns a google.protobuf.VAlue with its value set to val.
func StringValue(val string) *structpb.Value {
	return &structpb.Value{
		Kind: &structpb.Value_StringValue{
			StringValue: val,
		},
	}
}

// BoolValue returns a google.protobuf.Value with its value set to val.
func BoolValue(val bool) *structpb.Value {
	return &structpb.Value{
		Kind: &structpb.Value_BoolValue{
			BoolValue: val,
		},
	}
}

// Double returns a google.protobuf.DoubleValue with its value set to val
func Double(val float64) *wrappers.DoubleValue {
	return &wrappers.DoubleValue{Value: val}
}

// Int64 returns a google.protobuf.Int64Value with its value set to val
func Int64(val int64) *wrappers.Int64Value {
	return &wrappers.Int64Value{Value: val}
}

// Int32 returns a google.protobuf.Int32Value with its value set to val
func Int32(val int32) *wrappers.Int32Value {
	return &wrappers.Int32Value{Value: val}
}

// Uint64 returns a google.protobuf.UInt64Value with its value set to val.
func Uint64(val uint64) *wrappers.UInt64Value {
	return &wrappers.UInt64Value{Value: val}
}

// Uint32 returns a google.protobuf.UInt32Value with its value set to val.
func Uint32(val uint32) *wrappers.UInt32Value {
	return &wrappers.UInt32Value{Value: val}
}

// Bool returns  a google.protobuf.BoolValue with its value set to val.
func Bool(val bool) *wrappers.BoolValue {
	return &wrappers.BoolValue{Value: val}
}

// String returns a google.protobuf.StringValue with its value set to val.
func String(val string) *wrappers.StringValue {
	return &wrappers.StringValue{Value: val}
}
