package client

import (
	"context"
	"net/http"

	"github.com/pkg/errors"
	"github.com/twitchtv/twirp"

	rpc "code.justin.tv/video/viewcount-api/internal/rpc/viewcounts"
)

type Config struct {
	URL        string
	HTTPClient *http.Client
}

type Client struct {
	config      *Config
	twirpClient rpc.Viewcounts

	persistentHeaders http.Header
}

func New(c *Config) (*Client, error) {
	// setup http client
	if c.HTTPClient == nil {
		c.HTTPClient = &http.Client{}
	}

	// constant headers
	headers := make(http.Header)
	headers.Set(HeaderHost, getHostname())
	headers.Set(HeaderIP, getIP())

	// client
	client := &Client{
		config:            c,
		twirpClient:       rpc.NewViewcountsProtobufClient(c.URL, c.HTTPClient),
		persistentHeaders: headers,
	}

	return client, nil
}

// API
func (c *Client) ForStream(ctx context.Context, streamID uint64, requester string) (*Viewcount, error) {
	headers := c.persistentHeaders.Clone()
	headers.Add(HeaderID, requester)

	ctx, err := twirp.WithHTTPRequestHeaders(ctx, headers)
	if err != nil {
		return nil, errors.Wrap(err, "twirp error setting headers")
	}

	input := &rpc.StreamID{
		Id: streamID,
	}

	views, err := c.twirpClient.ForStream(ctx, input)
	if err != nil {
		return nil, errors.Wrap(err, "failed call to viewcountAPI")
	}

	v := &Viewcount{
		Count:           views.GetCount(),
		UnfilteredCount: views.GetCountUnfiltered(),
	}

	return v, nil
}

func (c *Client) ForAllStreams(ctx context.Context, requester string) (IntGroupedViewcounts, error) {
	headers := c.persistentHeaders.Clone()
	headers.Add(HeaderID, requester)

	ctx, err := twirp.WithHTTPRequestHeaders(ctx, headers)
	if err != nil {
		return nil, errors.Wrap(err, "twirp error setting headers")
	}

	input := &rpc.Empty{}

	streamViewcounts, err := c.twirpClient.ForAllStreams(ctx, input)
	if err != nil {
		return IntGroupedViewcounts{}, errors.Wrap(err, "failed call to viewcountAPI")
	}

	gvc := IntGroupedViewcounts{}
	for sid, vc := range streamViewcounts.Streams {
		gvc[sid] = &Viewcount{
			Count:           vc.GetCount(),
			UnfilteredCount: vc.GetCountUnfiltered(),
		}
	}

	return gvc, nil
}

func (c *Client) ForChannel(ctx context.Context, cid uint64, requester string) (*Viewcount, error) {
	headers := c.persistentHeaders.Clone()
	headers.Add(HeaderID, requester)

	ctx, err := twirp.WithHTTPRequestHeaders(ctx, headers)
	if err != nil {
		return nil, errors.Wrap(err, "twirp error setting headers")
	}

	input := &rpc.ChannelID{
		Id: cid,
	}

	views, err := c.twirpClient.ForChannel(ctx, input)
	if err != nil {
		return nil, errors.Wrap(err, "failed call to viewcountAPI")
	}

	v := &Viewcount{
		Count:           views.GetCount(),
		UnfilteredCount: views.GetCountUnfiltered(),
	}

	return v, nil
}

func (c *Client) ForAllChannels(ctx context.Context, requester string) (IntGroupedViewcounts, error) {
	headers := c.persistentHeaders.Clone()
	headers.Add(HeaderID, requester)

	ctx, err := twirp.WithHTTPRequestHeaders(ctx, headers)
	if err != nil {
		return nil, errors.Wrap(err, "twirp error setting headers")
	}

	input := &rpc.Empty{}

	channelViewcounts, err := c.twirpClient.ForAllChannels(ctx, input)
	if err != nil {
		return IntGroupedViewcounts{}, errors.Wrap(err, "failed call to viewcountAPI")
	}

	gvc := IntGroupedViewcounts{}
	for cid, vc := range channelViewcounts.Channels {
		gvc[cid] = &Viewcount{
			Count:           vc.GetCount(),
			UnfilteredCount: vc.GetCountUnfiltered(),
		}
	}

	return gvc, nil
}

func (c *Client) ForAllCustomers(ctx context.Context) (StringGroupedViewcounts, error) {
	ctx, err := twirp.WithHTTPRequestHeaders(ctx, c.persistentHeaders)
	if err != nil {
		return nil, errors.Wrap(err, "twirp error setting headers")
	}

	input := &rpc.Empty{}

	customerViewcounts, err := c.twirpClient.AllCustomerTotals(ctx, input)
	if err != nil {
		return StringGroupedViewcounts{}, errors.Wrap(err, "failed call to viewcountAPI")
	}

	gvc := StringGroupedViewcounts{}
	for cid, vc := range customerViewcounts.Customers {
		gvc[cid] = &Viewcount{
			Count:           vc.GetCount(),
			UnfilteredCount: vc.GetCountUnfiltered(),
		}
	}

	return gvc, nil
}

func (c *Client) ForAllStreamsByPOP(ctx context.Context, requester string) (POPGroupedViewcounts, error) {
	headers := c.persistentHeaders.Clone()
	headers.Add(HeaderID, requester)

	ctx, err := twirp.WithHTTPRequestHeaders(ctx, headers)
	if err != nil {
		return nil, errors.Wrap(err, "twirp error setting headers")
	}

	input := &rpc.Empty{}

	popViewcounts, err := c.twirpClient.ForAllStreamsByPop(ctx, input)
	if err != nil {
		return POPGroupedViewcounts{}, errors.Wrap(err, "failed call to viewcountAPI")
	}

	pgvc := POPGroupedViewcounts{}
	for pop, gvc := range popViewcounts.Pops {
		if _, ok := pgvc[pop]; !ok {
			pgvc[pop] = IntGroupedViewcounts{}
		}

		for sid, vc := range gvc.Streams {
			pgvc[pop][sid] = &Viewcount{
				Count:           vc.GetCount(),
				UnfilteredCount: vc.GetCountUnfiltered(),
			}
		}
	}

	return pgvc, nil
}

func (c *Client) ForAllStreamsByPopNode(ctx context.Context, requester string) (NodeGroupedViewcounts, error) {
	headers := c.persistentHeaders.Clone()
	headers.Add(HeaderID, requester)

	ctx, err := twirp.WithHTTPRequestHeaders(ctx, headers)
	if err != nil {
		return nil, errors.Wrap(err, "twirp error setting headers")
	}

	input := &rpc.Empty{}

	nodeViewcounts, err := c.twirpClient.ForAllStreamsByPopNode(ctx, input)
	if err != nil {
		return NodeGroupedViewcounts{}, errors.Wrap(err, "failed call to viewcountAPI")
	}

	ngvc := NodeGroupedViewcounts{}
	for pop, pgvc := range nodeViewcounts.Pops {
		if _, ok := ngvc[pop]; !ok {
			ngvc[pop] = map[string]IntGroupedViewcounts{}
		}

		for node, gvc := range pgvc.Nodes {
			if _, ok := ngvc[pop][node]; !ok {
				ngvc[pop][node] = IntGroupedViewcounts{}
			}

			for sid, vc := range gvc.Streams {
				ngvc[pop][node][sid] = &Viewcount{
					Count:           vc.GetCount(),
					UnfilteredCount: vc.GetCountUnfiltered(),
				}
			}
		}
	}

	return ngvc, nil
}
