/**
 * This script deploys your current branch to the gh-pages branch.
 * It automatically tries to figure out what your deployed URL should
 * be based on your current branch!
 */

const { execSync } = require('child_process');
const ghpages = require('gh-pages');
const report = require('yurnalist');
const Confirm = require('prompt-confirm');

run();

async function run() {
  printBigMessage(
    `This will upload your site to a Github pages URL to share internally. You can re-upload anytime by running this command from the same branch.\n\nBefore proceeding:\n    1. Connect to the Twitch VPN \n    2. Your Github SSH credentials must be configured.\n    3. Make a branch (such as 'bc-cool-demo') and commit your code!`
  );
  await confirmOrAbort('Start Deployment?');

  /**
   * Setup
   */
  execSync('git fetch origin');
  const baseUrl = getBaseUrl();
  const currentBranch = getSlugFromBranch();
  const targetUrl = `${baseUrl}${currentBranch}/`;

  report.step(1, 3, 'Pre-flight checks');

  /**
   * Preflight Checks
   */
  if (currentBranch === 'master') {
    exitWithError(
      'You are currently on the "master" branch. \nIn order to deploy your page, please create a new branch and give it a unique name. \n\nExample: "git checkout -b bc-cool-project"'
    );
  }

  if (currentBranch === 'static') {
    exitWithError("The branch name 'static' is not allowed. Please make a new branch.");
  }

  if (currentBranch === '') {
    exitWithError(
      "Could not detect your git branch name. This shouldn't happen; please reach out for support or you can manually deploy to a folder in the gh-pages branch."
    );
  }

  if (hasUncommitedChanges()) {
    printBigMessage(() =>
      report.warn(
        "You have local changes not yet committed to git. Commit your changes and run 'git push' (you can deploy anyways, but people will not be able to see your source code until you commit and push it)"
      )
    );
    await confirmOrAbort('Continue with deployment?');
  }

  if (hasUnsyncedCommits(currentBranch)) {
    printBigMessage(() =>
      report.warn(
        "Your local branch is not in sync with Github. Run 'git pull' and/or 'git push' (you can deploy anyways, but people will not be able to see your source code until you commit and push it)"
      )
    );
    await confirmOrAbort('Continue with deployment?');
  }

  /**
   * Confirm overwite existing site?
   */
  const history = getPreviousDeploys(currentBranch);

  if (history.length > 0) {
    printBigMessage(() => {
      report.warn(`This URL is already hosting something! \n\nPublish history:`);
      report.list(
        'publish history',
        history.map(row => `${row[0]} by ${row[1]} <${row[2]}>`)
      );
      process.stdout.write('\n');
      process.stdout.write(`Existing site: \n${targetUrl}\n`);
    });

    await confirmOrAbort('Overwrite and replace the existing site?');
  }

  // TODO: Check if you have push permission and if not prompt on how to request it!

  /**
   * Deploy it!
   */

  // Set homepage URL for the build
  process.env.PUBLIC_URL = targetUrl;
  process.env.REACT_APP_GIT_BUILD = report.step(2, 3, 'Building the site with yarn build');

  execSync('react-scripts build');

  report.step(3, 3, 'Uploading to Github with gh-pages');

  // 2 deploy
  ghpages.publish(
    'build',
    {
      message: `Deploy ${currentBranch}`,
      dest: currentBranch
    },
    () => {
      printBigMessage(() => report.success(`Published URL: \n${targetUrl}`));
    }
  );
}

function getBaseUrl() {
  return execSync('git ls-remote --get-url')
    .toString()
    .trim()
    .replace(':', '/pages/')
    .replace('git@', 'https://')
    .replace('.git', '/');
}

function getSlugFromBranch() {
  // TODO: Handle http vs git urls
  return execSync('git symbolic-ref --short HEAD')
    .toString()
    .trim()
    .toLowerCase();
}

function getPreviousDeploys(dir) {
  // 1. See if the dir is presently empty in gh-pages branch
  const dirExists = execSync(`git ls-tree -d origin/gh-pages:${dir} 2> /dev/null || echo ''`)
    .toString()
    .trim();

  if (dirExists.length === 0) {
    return [];
  }

  // 2. Get the commit history for the dir in gh-pages branch
  const previousDeployedBy = execSync(`git log origin/gh-pages --max-count=5 --format=%cr,%cn,%ce -- ${dir}`)
    .toString()
    .trim();

  if (previousDeployedBy.length === 0) {
    return [];
  }

  return previousDeployedBy.split('\n').map(v => v.split(','));
}

async function confirmOrAbort(message) {
  const userConfirmed = await new Confirm(message).run();

  if (!userConfirmed) {
    report.info('No changes have been made.');
    process.exit(0);
  }
}

function hasUncommitedChanges() {
  const result = execSync('git status --porcelain')
    .toString()
    .trim();

  return result.length > 0;
}

function hasUnsyncedCommits(branch) {
  // 1. Check if is published to remote
  const refMatch = execSync(`git show-ref refs/remotes/origin/${branch} || echo ''`)
    .toString()
    .trim();

  if (refMatch.length === 0) {
    return true;
  }

  // 2. Check if remote and local is out of sync
  const diff = execSync(`git rev-list --left-right --count ${branch}...origin/${branch}`)
    .toString()
    .trim()
    .split('\t');

  return 0 !== parseInt(diff[0]) + parseInt(diff[1]);
}

function printBigMessage(message) {
  process.stdout.write('\n');
  process.stdout.write(`- - - - - - - - - - - - - - - - - - - -\n`);
  if (typeof message === 'function') {
    message();
  } else {
    process.stdout.write(message + '\n');
  }
  process.stdout.write(`- - - - - - - - - - - - - - - - - - - -\n`);
  process.stdout.write('\n');
}

function exitWithError(message) {
  process.stdout.write(`\n#######################################\n`);
  report.error(message);
  process.stdout.write(`#######################################\n\n`);
  process.exit(0);
}
