import React from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  Display,
  Layout,
  Modal,
  ModalSize,
  ModalHeader,
  ModalBody,
  ModalFooter,
  SVGAsset,
  VerticalNavigation,
  VerticalNavigationItem,
  ButtonState
} from 'twitch-core-ui';
import Overlay from './overlay';
import { useWindowDimensions } from '../hooks/useWindowDimensions';
import { MODAL_HEADER_AND_FOOTER_HEIGHT } from '../utils/config';
import './styles/Form.scss';

interface ModalField {
  headerName: string;
  validFields: boolean;
}

interface Props {
  children: React.ReactChild;
  isModalOpen: boolean;
  setIsModalOpen: React.Dispatch<React.SetStateAction<boolean>>;
  modalTitle: string;
  modalFields: ModalField[];
  activeFormPage: number;
  setActiveFormPage: React.Dispatch<React.SetStateAction<number>>;
  submitForm: () => void;
  isFormSubmitting?: boolean;
  closeModal?: () => void;
  isHeader?: boolean;
  isEditing?: boolean;
}

function AtlasModal({
  children,
  isModalOpen,
  setIsModalOpen,
  modalTitle,
  modalFields,
  activeFormPage,
  setActiveFormPage,
  submitForm,
  isFormSubmitting = false,
  closeModal = undefined,
  isHeader = false
}: Props) {
  const { height } = useWindowDimensions();

  return (
    isModalOpen && (
      <Overlay onClickOut={closeModal ? closeModal : () => setIsModalOpen(false)} isHeader={isHeader}>
        <Layout className={`modal--wrapper ${modalFields.length > 1 ? '' : 'condensed'}`}>
          <Modal size={modalFields.length > 1 ? ModalSize.Large : ModalSize.Medium}>
            <Layout className="modal--header">
              <ModalHeader
                title={modalTitle}
                closeButton={{
                  'aria-label': 'Close Modal',
                  onClick: closeModal ? closeModal : () => setIsModalOpen(false)
                }}></ModalHeader>
            </Layout>
            <SimpleBarReact style={{ maxHeight: height < 800 ? `calc(100vh - ${MODAL_HEADER_AND_FOOTER_HEIGHT})` : 660 }}>
              <ModalBody>
                <form onSubmit={submitForm}>
                  <Layout display={Display.Flex} padding={{ y: 1 }}>
                    {modalFields.length > 1 && (
                      <Layout className="modal--sidebar" flexShrink={0}>
                        <VerticalNavigation>
                          {modalFields.map((field, i) => {
                            const page = i + 1;
                            const goToPage = e => {
                              e.preventDefault();
                              setActiveFormPage(page);
                            };
                            return (
                              <Layout key={i} className="vertical-navigation--item">
                                <VerticalNavigationItem
                                  iconAsset={activeFormPage <= page ? SVGAsset[`Smallroman${page}`] : SVGAsset.Check}
                                  onClick={goToPage}
                                  selected={activeFormPage === page}
                                  disabled={page > activeFormPage && !modalFields[activeFormPage - 1].validFields}>
                                  {field.headerName}
                                </VerticalNavigationItem>
                              </Layout>
                            );
                          })}
                        </VerticalNavigation>
                      </Layout>
                    )}
                    <Layout className={modalFields.length > 1 ? 'modal--body' : ''} padding={{ left: modalFields.length > 1 ? 5 : 1, top: 1 }}>
                      {children}
                    </Layout>
                  </Layout>
                </form>
              </ModalBody>
            </SimpleBarReact>
            <ModalFooter
              primaryButtonProps={{
                children: modalFields.length === activeFormPage ? 'Submit' : 'Next',
                onClick: modalFields.length === activeFormPage ? () => submitForm() : () => setActiveFormPage(activeFormPage + 1),
                state: isFormSubmitting ? ButtonState.Loading : ButtonState.Default,
                disabled: isFormSubmitting || !modalFields[activeFormPage - 1].validFields
              }}
              secondaryButtonProps={{
                children: 'Cancel',
                state: isFormSubmitting ? ButtonState.Loading : ButtonState.Default,
                disabled: isFormSubmitting,
                onClick: closeModal ? closeModal : () => setIsModalOpen(false)
              }}
            />
          </Modal>
        </Layout>
      </Overlay>
    )
  );
}

export default AtlasModal;
