import React, { useState, useRef } from 'react';
import {
  CoreText,
  Layout,
  Display,
  AlignItems,
  SearchInput,
  Position,
  DropDownMenu,
  DropDownMenuItem,
  Transition,
  TransitionType,
  ZIndex
} from 'twitch-core-ui';
import ClickOutDetector from './click-out-detector';
import Loading from './loading';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import AvatarFallback from './avatar-fallback';
import useUpdateEffect from '../hooks/useUpdateEffect';
import './styles/Form.scss';
import { useLazyQuery } from '@apollo/client';
import { loader } from 'graphql.macro';
import { SearchUsersQueryInputs, SearchUsersQueryResults } from '../utils/gqlTypes';
import { Channel } from '../utils/types';
const SearchUsersQuery = loader('../gql/searchUsers.gql');

interface ChannelDropdownProps { 
  channels, 
  isChannelSearchFocused, 
  setIsChannelSearchFocused, 
  handleAddChannel: (e, channel: Channel) => void, 
}

function ChannelDropdown(props: ChannelDropdownProps) {
  const { channels, isChannelSearchFocused, setIsChannelSearchFocused, handleAddChannel } = props;
  const [searchQuery, setSearchQuery] = useState('');
  const [searchUsers, { data, loading, error }] = useLazyQuery<SearchUsersQueryResults, SearchUsersQueryInputs>(SearchUsersQuery)
  const timeoutRef = useRef(null);
  const users = data?.searchUsers?.edges;
  useUpdateEffect(() => {
    if (timeoutRef.current !== null) {
      clearTimeout(timeoutRef.current);
    }
    timeoutRef.current = setTimeout(() => {
      timeoutRef.current = null;
      searchUsers({
        variables: { userQuery: searchQuery },
      });
    }, 300);
    return () => {
      timeoutRef.current = false;
    }; /* eslint-disable react-hooks/exhaustive-deps */
  }, [searchQuery, searchUsers]);

  return (
    <ClickOutDetector onClickOut={() => setIsChannelSearchFocused(false)}>
      <SearchInput
        placeholder="Channel(s)"
        onFocus={() => setIsChannelSearchFocused(!isChannelSearchFocused)}
        onChange={e => setSearchQuery(e.currentTarget.value)}
      />
      <Layout position={Position.Relative} fullWidth zIndex={ZIndex.Above}>
        <Transition type={TransitionType.SlideOverTop} show={isChannelSearchFocused}>
          <DropDownMenu show={isChannelSearchFocused && !!searchQuery}>
            <div style={{ width: '100%' }}>
              {error && <ErrorAlert error={error} />}
              {loading ? (
                <Loading />
              ) : users ? (
                  users
                  .filter(user => user.node?.id && channels.every(selectedChannels => selectedChannels.channel_id !== user.node.id))
                  .map(user => {
                    const { id, login, profileImageURL } = user.node;
                    const channel = {
                      channel_id: parseInt(id),
                      channel_login: login,
                      profile_image: profileImageURL
                    };
                    return (
                      <DropDownMenuItem key={id} onClick={e => handleAddChannel(e, channel)}>
                        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                          <Layout margin={{ right: 1 }}>
                            <AvatarFallback profileImage={profileImageURL} login={login} size={24} />
                          </Layout>
                          <CoreText>{login}</CoreText>
                        </Layout>
                      </DropDownMenuItem>
                    );
                  })
              ) : (
                <NoResults results="channels" />
              )}
            </div>
          </DropDownMenu>
        </Transition>
      </Layout>
    </ClickOutDetector>
  );
}

export default ChannelDropdown;
