import React, { useState, useEffect } from 'react';
import Cookies from 'js-cookie';
import {
  CoreText,
  TextType,
  Layout,
  StyledLayout,
  FormGroup,
  Input,
  InputType,
  Select,
  Color,
  InFeatureNotification,
  NotificationType,
  Position,
  Transition,
  TransitionType,
  BorderRadius
} from 'twitch-core-ui';
import FormChannelsSelect from './form-channels-select';
import { ATLAS_BASE_URL } from '../utils/config';
import { contentCreatorTypeOptions } from '../utils/options';
import { useFetchPOST } from '../hooks/useFetch';
import { fetchWithTimeout } from '../utils/helpers';
import { writeStorage } from '@rehooks/local-storage';
import AtlasModal from './atlas-modal';
import MultiselectAccountManager from './multiselect-account-manager';
import useUpdateEffect from '../hooks/useUpdateEffect';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function ContentCreatorModal({ isModalOpen, setIsModalOpen, modalTitle, contentCreators, selectedContentCreator }) {
  // modal form state
  const [activeFormPage, setActiveFormPage] = useState(1);
  const [isFormSubmitting, setIsFormSubmitting] = useState(false);
  const [isValidPCC, setIsValidPCC] = useState(true);
  const [isPccAlertDisplayed, setIsPccAlertDisplayed] = useState(false);
  const [isPccEditError, setIsPccEditError] = useState(false);

  useUpdateEffect(() => {
    setTimeout(() => {
      setIsPccAlertDisplayed(false);
    }, 5000);
  }, [isPccAlertDisplayed]);

  const [amList] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-account-manager-list`, {
    eventtype: 1,
    limit: 1000,
    'req-cols': ['account_manager_id', 'account_manager_first_name', 'account_manager_last_name', 'account_manager_ldap_name'],
    sortdata: [{ ord: 0, col: 'account_manager_first_name', asc: true }]
  });

  // modal form data
  // hidden
  const [contentCreatorId, setContentCreatorId] = useState(0);
  // page 1
  const [contentCreatorName, setContentCreatorName] = useState('');
  const [isPccNameDuplicate, setIsPccNameDuplicate] = useState(false);
  const [pccNameErrorMessage, setPccNameErrorMessage] = useState('');
  const [contentCreatorType, setContentCreatorType] = useState('');
  const [accountManagers, setAccountManagers] = useState([]);

  // page 2
  const [channels, setChannels] = useState([]);

  useEffect(() => {
    if (selectedContentCreator === undefined) {
      setIsValidPCC(false);
      return;
    }

    const { premium_content_creator_id, premium_content_creator_name, premium_content_creator_type, am_ids, channels } = selectedContentCreator;

    setContentCreatorId(premium_content_creator_id || 0);
    setContentCreatorName(premium_content_creator_name || '');
    setContentCreatorType(premium_content_creator_type || '');
    setChannels(channels || []);

    let amArr = [];
    amList &&
      amList.data &&
      amList.data.items &&
      amList.data.items.forEach(am => {
        if (am_ids) {
          for (let amId of am_ids) {
            if (amId === am.account_manager_id) {
              amArr.push(am);
            }
          }
        }
      });
    setAccountManagers(amArr);
  }, [selectedContentCreator, amList]);

  const closeModal = () => {
    setIsModalOpen(false);
    setActiveFormPage(1);
    if (Object.keys(selectedContentCreator).length !== 0) {
      setContentCreatorId(0);
      setContentCreatorName('');
      setContentCreatorType('');
      setAccountManagers([]);
      setChannels([]);
    }
  };

  const submitForm = () => {
    if (modalTitle.includes('Add')) {
      const pccNameDuplicateList = contentCreators.filter(ccObj => ccObj.premium_content_creator_name === contentCreatorName);
      if (pccNameDuplicateList.length) {
        setActiveFormPage(1);
        setIsPccNameDuplicate(true);
        setPccNameErrorMessage('Premium Content Creator Name already exists. Please try a unique PCC Name.');
        return;
      }
    }

    let contentCreator = {
      premium_content_creator_id: contentCreatorId,
      premium_content_creator_name: contentCreatorName,
      premium_content_creator_type: contentCreatorType,
      is_active: true
    };

    let accountManagerIDs = [];
    accountManagers.forEach(am => {
      accountManagerIDs.push(am.account_manager_id);
    });

    let obj = {
      creator: contentCreator,
      channels: channels,
      am_ids: accountManagerIDs
    };

    const body = Object.keys(obj)
      .filter(e => obj[e] !== null)
      .reduce((o, e) => {
        o[e] = obj[e];
        return o;
      }, {});

    setIsFormSubmitting(true);
    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(body)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-premium-content-creator-with-channels`, options);
        const json = await response.json();
        console.log(`/edit-premium-content-creator-with-channels:`, json);
        closeModal();

        if (contentCreatorId === parseInt(localStorage.getItem(PCCID_LOCAL_STORAGE_KEY))) {
          writeStorage('productOwner', contentCreatorName);
        }

        if (modalTitle.includes('Add')) {
          writeStorage('productOwner', contentCreatorName);
          let newContentCreatorId = json.data;
          writeStorage('productOwnerID', newContentCreatorId);
        }
      } catch (error) {
        console.error(`/edit-premium-content-creator-with-channels:`, error);
        setIsPccEditError(true);
        closeModal();
        return;
      }
      setIsFormSubmitting(false);
      setIsPccAlertDisplayed(true);
    }
    fetchData();
  };

  const changeContentCreatorName = e => {
    setContentCreatorName(e.currentTarget.value);
    if (isPccNameDuplicate) {
      setPccNameErrorMessage('');
      setIsPccNameDuplicate(false);
    }
  };

  const renderModalBodyForm = () => {
    switch (activeFormPage) {
      case 1:
        return (
          <Layout>
            <Layout padding={{ bottom: 1 }}>
              <CoreText type={TextType.H3}>Content Creator Details</CoreText>
            </Layout>
            <StyledLayout padding={{ bottom: 3 }} margin={{ bottom: 3 }} borderBottom>
              <CoreText type={TextType.P} color={Color.Alt}>
                Provide details about the content creator and manage associated channels
              </CoreText>
            </StyledLayout>
            <Layout margin={{ bottom: 2 }}>
              <FormGroup label="Name" hint="Name this creator" required error={isPccNameDuplicate} errorMessage={pccNameErrorMessage}>
                <Input
                  type={InputType.Text}
                  placeholder="Creator Name"
                  error={isPccNameDuplicate}
                  value={contentCreatorName}
                  onChange={changeContentCreatorName}
                />
              </FormGroup>
            </Layout>
            <Layout margin={{ bottom: 2 }}>
              <FormGroup label="Type" hint="The type of Creator" required>
                <Select value={contentCreatorType} onChange={e => setContentCreatorType(e.currentTarget.value)}>
                  <option disabled value="">
                    Content Creator Type
                  </option>
                  {contentCreatorTypeOptions.map(type => {
                    return (
                      <option key={type} value={type}>
                        {type}
                      </option>
                    );
                  })}
                </Select>
              </FormGroup>
            </Layout>
            <Layout margin={{ bottom: 2 }}>
              <FormGroup label="Twitch Account Manager / Point of Contact" hint="This creator’s internal account manager or point of contact at Twitch">
                <MultiselectAccountManager
                  accountManagers={accountManagers}
                  setAccountManagers={setAccountManagers}
                  dropdownOptions={amList && amList.data && amList.data.items}
                />
              </FormGroup>
            </Layout>
          </Layout>
        );

      case 2:
        return (
          <Layout>
            <Layout padding={{ bottom: 1 }}>
              <CoreText type={TextType.H3}>Channel Management</CoreText>
            </Layout>
            <StyledLayout padding={{ bottom: 3 }} margin={{ bottom: 3 }} borderBottom>
              <CoreText type={TextType.P} color={Color.Alt}>
                List all the relevant channels that will be included in this contract
              </CoreText>
            </StyledLayout>
            <FormChannelsSelect channels={channels} setChannels={setChannels} />
          </Layout>
        );
    }
  };

  const contentCreatorFields = [
    { headerName: 'Content Creator Details', validFields: !!contentCreatorName && !!contentCreatorType },
    { headerName: 'Channel Management', validFields: true }
  ];

  if (isValidPCC) {
    return (
      <>
        <AtlasModal
          isModalOpen={isModalOpen}
          setIsModalOpen={setIsModalOpen}
          modalTitle={modalTitle}
          modalFields={contentCreatorFields}
          activeFormPage={activeFormPage}
          setActiveFormPage={setActiveFormPage}
          submitForm={submitForm}
          closeModal={closeModal}
          isFormSubmitting={isFormSubmitting}
          isHeader>
          {renderModalBodyForm()}
        </AtlasModal>
        {isPccAlertDisplayed ? (
          <StyledLayout className="pcc-alert" position={Position.Absolute} attachTop elevation={1} borderRadius={BorderRadius.Medium}>
            <Transition type={TransitionType.SlideOverTop} show={isPccAlertDisplayed}>
              <InFeatureNotification
                type={isPccEditError ? NotificationType.Error : NotificationType.Success}
                message={{
                  title: `PCC ${modalTitle.includes('Edit') ? 'Edit' : 'Create'} ${isPccEditError ? 'Fail' : 'Success'}`,
                  description: `Premium Content Creator ${modalTitle.includes('Edit') ? 'edit' : 'creat'} ${isPccEditError ? 'failed' : 'successful'}`
                }}
                closeButton={{
                  'aria-label': 'Close Modal',
                  onClick: () => setIsPccAlertDisplayed(false)
                }}
              />
            </Transition>
          </StyledLayout>
        ) : null}
      </>
    );
  } else {
    return (
      <Layout className="pcc-alert" position={Position.Absolute} attachTop>
        <InFeatureNotification
          type={NotificationType.Error}
          message={{
            title: 'Invalid Premium Content Creator',
            description: 'Looks like the selected Premium Content Creator no longer exists. Please try selecting another Premium Content Creator.'
          }}
          closeButton={{
            'aria-label': 'Close Modal',
            onClick: () => setIsValidPCC(true)
          }}
        />
      </Layout>
    );
  }
}

export default ContentCreatorModal;
