import React, { useState, useEffect } from 'react';
import { useHistory, useParams } from 'react-router-dom';
import Cookies from 'js-cookie';
import {
  Layout,
  CoreText,
  TextType,
  StyledLayout,
  FormGroup,
  FormGroupOrientation,
  Input,
  InputType,
  Select,
  Button,
  ButtonType,
  Display,
  JustifyContent,
  ButtonState,
  Background,
  BorderRadius,
  AlignItems,
  CheckBox,
  Toggle,
  Transition,
  TransitionType,
  TextArea
} from 'twitch-core-ui';
import FormChannelsSelect from './form-channels-select';
import FormRadioOption from './form-radio-option';
import FormRadioInputOption from './form-radio-input-option';
import DatePicker from './date-picker';
import MultiselectAccountManager from './multiselect-account-manager';
import Loading from './loading';
import ErrorAlert from './error-alert';
import { ATLAS_BASE_URL } from '../utils/config';
import { adOptions, tier1Options, tier2Options, tier3Options, checkboxOptions } from '../utils/options';
import { useFetchPOST, useFetchGET } from '../hooks/useFetch';
import { useLocalStorage } from '@rehooks/local-storage';
import { fetchWithTimeout } from '../utils/helpers';
import { getMomentDate } from '../utils/dateTimeHelpers';
import './styles/Form.scss';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function ContractsEdit() {
  const history = useHistory();
  let { id } = useParams<{ id: string }>();
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  let options = {
    limit: 1,
    page: 0,
    eventtype: 1,
    'req-cols': ['*'],
    filters: [
      { 'filter-type': 1, key: 'contract_id', v1: id },
      { 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }
    ]
  };
  const [data, loading] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-contract-list`, options);
  const [isFormSubmitting, setIsFormSubmitting] = useState(false);
  const [error, setError] = useState('');

  const [selectedContentCreatorID] = useLocalStorage('productOwnerID', '0');
  const [contentCreatorChannels] = useFetchGET(`${ATLAS_BASE_URL}/event/get-pcc-channel-data/${selectedContentCreatorID}`);

  const [creatorOptions] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-premium-content-creator-list`, {
    eventtype: 1,
    'req-cols': ['premium_content_creator_id', 'premium_content_creator_name'],
    sortdata: [{ ord: 0, col: 'premium_content_creator_name', asc: true }]
  });

  const [amList] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-account-manager-list`, {
    eventtype: 1,
    limit: 1000,
    'req-cols': ['account_manager_id', 'account_manager_first_name', 'account_manager_last_name', 'account_manager_ldap_name'],
    sortdata: [{ ord: 0, col: 'account_manager_first_name', asc: true }]
  });

  // modal form data
  // hidden
  const [contractId, setContractId] = useState('0');
  const [splitLocked, setSplitLocked] = useState(false);
  // page 1
  const [contractName, setContractName] = useState('');
  const [creator, setCreator] = useState('0');
  const [accountManagers, setAccountManagers] = useState([]);
  const [startDate, setStartDate] = useState(null);
  const [endDate, setEndDate] = useState(null);
  const [type, setType] = useState('');
  const [exclusivity, setExclusivity] = useState('');
  const [prerolls, setPrerolls] = useState('');
  const [sponsorship, setSponsorship] = useState('');
  const [adDensity, setAdDensity] = useState('');
  // page 2
  const [revenueShare, setRevenueShare] = useState(false);
  const [adType, setAdType] = useState('');
  const [tier1, setTier1] = useState('');
  const [tier1Value, setTier1Value] = useState(0);
  const [tier2, setTier2] = useState('');
  const [tier2Value, setTier2Value] = useState(0);
  const [tier3, setTier3] = useState('');
  const [tier3Value, setTier3Value] = useState(0);
  const [bits, setBits] = useState('');
  const [subscriptions, setSubscriptions] = useState('');
  // page 3
  const [channels, setChannels] = useState([]);
  // page 4
  const [notes, setNotes] = useState('');
  const [documentLink, setdocumentLink] = useState('');
  const [autoRenew, setAutoRenew] = useState(false);

  useEffect(() => {
    if (data !== undefined && data.data !== undefined) {
      const {
        contract_id,
        contract_name,
        premium_content_creator_id,
        account_managers,
        start_date,
        end_date,
        contract_type,
        exclusivity_enabled,
        ad_prerolls_enabled,
        sponsorship_enabled,
        revenue_share_enabled,
        ad_revenue_contract_type,
        ad_revenue_share_type_tier_1,
        ad_revenue_share_amount_tier_1,
        ad_revenue_share_type_tier_2,
        ad_revenue_share_amount_tier_2,
        ad_revenue_share_type_tier_3,
        ad_revenue_share_amount_tier_3,
        bits_revenue_share_percent,
        subs_revenue_share_percent,
        additional_notes,
        document_link,
        auto_renew,
        channels,
        ad_density_per_hour
      } = data.data.items[0];

      setContractId(contract_id || '0');
      setContractName(contract_name || '');
      setCreator(premium_content_creator_id || '0');
      let amArr = [];
      amList &&
        amList.data &&
        amList.data.items.forEach(am => {
          if (account_managers) {
            for (let amId of account_managers) {
              if (amId.account_manager_id === am.account_manager_id) {
                amArr.push(am);
              }
            }
          }
        });
      setAccountManagers(amArr);
      setStartDate(start_date ? new Date(getMomentDate(start_date)) : null);
      setEndDate(end_date ? new Date(getMomentDate(end_date)) : null);
      setType(contract_type || '');
      setExclusivity(typeof exclusivity_enabled === 'boolean' ? `${exclusivity_enabled}` : '');
      setPrerolls(typeof ad_prerolls_enabled === 'boolean' ? `${ad_prerolls_enabled}` : '');
      setSponsorship(typeof sponsorship_enabled === 'boolean' ? `${sponsorship_enabled}` : '');
      setRevenueShare(revenue_share_enabled || false);
      setAdType(ad_revenue_contract_type || '');
      if (ad_revenue_contract_type !== 'Standard') {
        setSplitLocked(true);
      }
      setTier1(ad_revenue_share_type_tier_1 || '');
      setTier1Value(ad_revenue_share_amount_tier_1 || 0);
      setTier2(ad_revenue_share_type_tier_2 || '');
      setTier2Value(ad_revenue_share_amount_tier_2 || 0);
      setTier3(ad_revenue_share_type_tier_3 || '');
      setTier3Value(ad_revenue_share_amount_tier_3 || 0);
      setBits(bits_revenue_share_percent == null ? '' : bits_revenue_share_percent.toString());
      setSubscriptions(subs_revenue_share_percent == null ? '' : subs_revenue_share_percent.toString());
      setAdDensity(ad_density_per_hour == null ? '' : ad_density_per_hour.toString());
      setdocumentLink(document_link || '');
      setNotes(additional_notes || '');
      setAutoRenew(auto_renew || false);
      let defaultChannels = [];
      if (contentCreatorChannels && contentCreatorChannels.data) {
        contentCreatorChannels.data.forEach(channel => {
          defaultChannels.push(channel);
        });
      }
      setChannels(channels || defaultChannels);
    } /* eslint-disable react-hooks/exhaustive-deps */
  }, [data]);

  const submitForm = () => {
    let accountManagerIDs = [];
    accountManagers.forEach(am => {
      accountManagerIDs.push(am.account_manager_id);
    });
    let contract = {
      contract_id: contractId,
      contract_name: contractName,
      premium_content_creator_id: creator === '0' ? selectedContentCreatorID : creator,
      start_date: startDate.toISOString(),
      end_date: endDate.toISOString(),
      contract_type: type,
      exclusivity_enabled: exclusivity,
      ad_prerolls_enabled: prerolls,
      sponsorship_enabled: sponsorship,
      revenue_share_enabled: revenueShare,
      ad_revenue_contract_type: adType,
      ad_revenue_share_type_tier_1: tier1,
      ad_revenue_share_amount_tier_1: tier1Value,
      ad_revenue_share_type_tier_2: tier2,
      ad_revenue_share_amount_tier_2: tier2Value,
      ad_revenue_share_type_tier_3: tier3,
      ad_revenue_share_amount_tier_3: tier3Value,
      bits_revenue_share_percent: bits,
      subs_revenue_share_percent: subscriptions,
      ad_density_per_hour: adDensity,
      additional_notes: notes,
      document_link: documentLink,
      auto_renew: autoRenew,
      is_active: true
    };

    let obj = {
      contract: contract,
      account_manager_ids: accountManagerIDs,
      channels: channels
    };

    const body = Object.keys(obj)
      .filter(e => obj[e] !== null)
      .reduce((o, e) => {
        o[e] = obj[e];
        return o;
      }, {});

    setIsFormSubmitting(true);
    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(body)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-contract-with-channels`, options);
        const json = await response.json();
        console.log(`/edit-contract-with-channels:`, json);

        if (json.success) {
          history.push('/contracts');
        } else {
          setError(!!json.error_message ? json.error_message : 'Edit Contract with Channels request not successful');
          setIsFormSubmitting(false);
        }
      } catch (err) {
        console.error(`/edit-contract-with-channels:`, err);
        setError(err);
        setIsFormSubmitting(false);
      }
    }
    fetchData();
  };

  const handleContractTypeChange = contractType => {
    setAdType(contractType);
    if (contractType === 'Standard') {
      setSplitLocked(true);
      setTier1('cpm');
      setTier1Value(3.5);
      setTier2('cpm');
      setTier2Value(3.5);
      setTier3('% of net');
      setTier3Value(50);
    } else if (contractType === 'Premium') {
      setSplitLocked(true);
      setTier1('cpm');
      setTier1Value(5);
      setTier2('cpm');
      setTier2Value(3.5);
      setTier3('% of net');
      setTier3Value(50);
    } else if (contractType === 'Equal Split') {
      setSplitLocked(true);
      setTier1('% of net');
      setTier1Value(50);
      setTier2('% of net');
      setTier2Value(50);
      setTier3('% of net');
      setTier3Value(50);
    } else if (contractType === 'Custom') {
      setSplitLocked(false);
    }
  };

  const isFormValid =
    !!contractName &&
    !!accountManagers &&
    !!startDate &&
    !!endDate &&
    !!type &&
    !!exclusivity &&
    !!prerolls &&
    !!sponsorship &&
    !!adDensity &&
    (!!revenueShare ? !!adOptions && !!bits && !!subscriptions : true);

  return (
    <Layout className="form--wrapper" margin="auto">
      <Layout padding={{ bottom: 1 }}>
        <CoreText type={TextType.H3}>Edit Contract</CoreText>
      </Layout>

      {loading ? (
        <Loading />
      ) : (
        <>
          <Layout margin={{ bottom: 2 }}>
            <CoreText type={TextType.H4}>Contract Details</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Name" hint="Name your Contract" required>
                <Input type={InputType.Text} placeholder="Contract Name" value={contractName} onChange={e => setContractName(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            {selectedContentCreatorID === '0' && (
              <StyledLayout padding={2} borderBottom>
                <FormGroup
                  orientation={FormGroupOrientation.Horizontal}
                  label="Creator"
                  hint="Which premium content creator is this contract with? Contracts require the PCC have an AM assignment">
                  <Select value={creator} onChange={e => setCreator(e.currentTarget.value)}>
                    <option disabled value="0">
                      Select Creator
                    </option>
                    {creatorOptions &&
                      creatorOptions.data &&
                      creatorOptions.data.items.map(option => (
                        <option key={option.premium_content_creator_id} value={option.premium_content_creator_id}>
                          {option.premium_content_creator_name}
                        </option>
                      ))}
                  </Select>
                </FormGroup>
              </StyledLayout>
            )}
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Account Manager" hint="Account manager assigned to this contract" required>
                <MultiselectAccountManager
                  accountManagers={accountManagers}
                  setAccountManagers={setAccountManagers}
                  dropdownOptions={amList && amList.data && amList.data.items}
                />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Start &amp; End Date" required>
                <Layout display={Display.Flex} alignItems={AlignItems.Center} fullWidth>
                  <Layout margin={{ right: 2 }} fullWidth>
                    <DatePicker defaultDate={startDate} onChange={setStartDate} maxDate={endDate} />
                  </Layout>
                  <Layout fullWidth>
                    <DatePicker defaultDate={endDate} onChange={setEndDate} minDate={startDate} position="right" />
                  </Layout>
                </Layout>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Auto-Renewal Contract">
                <CheckBox
                  label="This contract is automatically extended for another year at the end of the current period"
                  checked={autoRenew}
                  onChange={e => setAutoRenew(e.currentTarget.checked)}
                />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Type" hint="The type of contract" required>
                <Select value={type} onChange={e => setType(e.currentTarget.value)}>
                  <option disabled value="">
                    Select Type
                  </option>
                  {['Revenue Share', 'Minimum Guarantee', 'License Fee', '(Other)'].map(option => (
                    <option key={option} value={option}>
                      {option}
                    </option>
                  ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="Exclusivity"
                hint="This contract includes the exclusive right to broadcast contractual content"
                required>
                <Select value={exclusivity} onChange={e => setExclusivity(e.currentTarget.value)} required>
                  <option disabled value="">
                    Select Exclusivity
                  </option>
                  {checkboxOptions.map(option => (
                    <option key={option.value} value={option.value}>
                      {option.label}
                    </option>
                  ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="Prerolls Active"
                hint="PCC O&amp;O Channels will have preroll ads active by default"
                required>
                <Select value={prerolls} onChange={e => setPrerolls(e.currentTarget.value)} required>
                  <option disabled value="">
                    Select Prerolls
                  </option>
                  {checkboxOptions.map(option => (
                    <option key={option.value} value={option.value}>
                      {option.label}
                    </option>
                  ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="Sponsorship"
                hint="Twitch has the right to sell sponsorships against the contractual content"
                required>
                <Select value={sponsorship} onChange={e => setSponsorship(e.currentTarget.value)} required>
                  <option disabled value="">
                    Select Sponsorship
                  </option>
                  {checkboxOptions.map(option => (
                    <option key={option.value} value={option.value}>
                      {option.label}
                    </option>
                  ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Ad Density" hint="Number of Ads minutes per hour" required>
                <Input type={InputType.Number} placeholder="Ad min per hour" value={adDensity} onChange={e => setAdDensity(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          <Layout margin={{ bottom: 2, top: 5 }}>
            <CoreText type={TextType.H3}>Revenue &amp; Ads</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Revenue Share" hint="Will this contract include revenue sharing">
                <Layout className="toggle--wrapper">
                  <Toggle checked={revenueShare} onChange={e => setRevenueShare(e.currentTarget.checked)} />
                </Layout>
              </FormGroup>
            </StyledLayout>
            <div className={revenueShare ? 'ads--wrapper' : 'hide'}>
              <Transition type={TransitionType.SlideOverTop} show={revenueShare}>
                <StyledLayout padding={{ y: 2, left: 4, right: 2 }} borderBottom>
                  <FormGroup
                    orientation={FormGroupOrientation.Horizontal}
                    label="Ads"
                    hint="Select the agreement which will determine the advertising revenue split"
                    required={!!revenueShare}>
                    {adOptions.map((option, i) => (
                      <FormRadioOption key={i} option={option} type={adType} setType={handleContractTypeChange} />
                    ))}
                  </FormGroup>
                </StyledLayout>
                <StyledLayout padding={{ y: 2, left: 4, right: 2 }} borderBottom>
                  <FormGroup
                    orientation={FormGroupOrientation.Horizontal}
                    label="Tier 1"
                    hint="Select and define an amount for either CPM or Percentage for this tier">
                    {tier1Options.map((option, i) => (
                      <FormRadioInputOption
                        key={i}
                        option={option}
                        type={tier1}
                        setType={setTier1}
                        amount={tier1Value}
                        setAmount={setTier1Value}
                        disabled={splitLocked}
                      />
                    ))}
                  </FormGroup>
                </StyledLayout>
                <StyledLayout padding={{ y: 2, left: 4, right: 2 }} borderBottom>
                  <FormGroup
                    orientation={FormGroupOrientation.Horizontal}
                    label="Tier 2"
                    hint="Select and define an amount for either CPM or Percentage for this tier">
                    {tier2Options.map((option, i) => (
                      <FormRadioInputOption
                        key={i}
                        option={option}
                        type={tier2}
                        setType={setTier2}
                        amount={tier2Value}
                        setAmount={setTier2Value}
                        disabled={splitLocked}
                      />
                    ))}
                  </FormGroup>
                </StyledLayout>
                <StyledLayout padding={{ y: 2, left: 4, right: 2 }} borderBottom>
                  <FormGroup
                    orientation={FormGroupOrientation.Horizontal}
                    label="Tier 3"
                    hint="Select and define an amount for either CPM or Percentage for this tier">
                    {tier3Options.map((option, i) => (
                      <FormRadioInputOption
                        key={i}
                        option={option}
                        type={tier3}
                        setType={setTier3}
                        amount={tier3Value}
                        setAmount={setTier3Value}
                        disabled={splitLocked}
                      />
                    ))}
                  </FormGroup>
                </StyledLayout>
              </Transition>
            </div>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="Bits"
                hint="Percentage of bits revenue awarded to the PCC"
                required={!!revenueShare}>
                <Input type={InputType.Number} placeholder="XX.XX %" max={100} value={bits} onChange={e => setBits(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="Subscriptions"
                hint="Percentage of subscription revenue awarded to the PCC"
                required={!!revenueShare}>
                <Input type={InputType.Number} placeholder="XX.XX %" max={100} value={subscriptions} onChange={e => setSubscriptions(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          <Layout margin={{ bottom: 2, top: 5 }}>
            <CoreText type={TextType.H3}>Channel Management</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormChannelsSelect channels={channels} setChannels={setChannels} channelType="date" />
            </StyledLayout>
          </StyledLayout>

          <Layout margin={{ bottom: 2, top: 5 }}>
            <CoreText type={TextType.H3}>Additional Notes</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Notes" hint="Any additional notes that are needed">
                <TextArea placeholder="Add additional notes here" value={notes} onChange={e => setNotes(e.currentTarget.value)} rows={10} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Document Link" hint="Link to a Google doc with more info">
                <Input type={InputType.Text} placeholder="Document Link" value={documentLink} onChange={e => setdocumentLink(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          {error && (
            <Layout margin={{ y: 3 }}>
              <ErrorAlert error={error} />
            </Layout>
          )}

          <Layout margin={{ y: 3 }} display={Display.Flex} justifyContent={JustifyContent.End}>
            <Layout margin={{ right: 1 }}>
              <Button
                type={ButtonType.Secondary}
                linkTo="/contracts"
                state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default}
                disabled={isFormSubmitting}>
                Cancel
              </Button>
            </Layout>
            <Button onClick={submitForm} state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default} disabled={isFormSubmitting || !isFormValid}>
              Submit
            </Button>
          </Layout>
        </>
      )}
    </Layout>
  );
}

export default ContractsEdit;
