import React, { useState } from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  CoreText,
  TextType,
  Layout,
  StyledLayout,
  Display,
  Background,
  AlignItems,
  SVG,
  SVGAsset,
  JustifyContent,
  FontSize,
  Button,
  ButtonSize,
  ButtonType,
  ButtonIcon,
  Table,
  TableHeader,
  TableHeading,
  TableBody,
  TableRow,
  TableCell,
  FontWeight,
  FlexWrap,
  Transition,
  TransitionType,
  Color,
  CoreLink,
  SVGType
} from 'twitch-core-ui';
import AvatarFallback from './avatar-fallback';
import { getMomentDate } from '../utils/dateTimeHelpers';
import './styles/Table.scss';

function ContractsList({ contracts, editContract }) {
  const [openContract, setOpenContract] = useState(null);

  const setActiveContract = contractId => {
    setOpenContract(contractId === openContract ? null : contractId);
  };

  return (
    contracts &&
    contracts.map(contract => {
      const {
        contract_id,
        contract_name,
        contract_type,
        account_managers,
        start_date,
        end_date,
        ad_revenue_contract_type,
        ad_revenue_share_amount_tier_1,
        ad_revenue_share_amount_tier_2,
        ad_revenue_share_amount_tier_3,
        ad_revenue_share_type_tier_1,
        ad_revenue_share_type_tier_2,
        ad_revenue_share_type_tier_3,
        bits_revenue_share_percent,
        subs_revenue_share_percent,
        ad_density_per_hour,
        exclusivity_enabled,
        ad_prerolls_enabled,
        sponsorship_enabled,
        channels,
        additional_notes,
        document_link
      } = contract;
      let accountManagerNames = [];
      if (account_managers != null) {
        account_managers.forEach(accountManager => {
          accountManagerNames.push(accountManager.account_manager_first_name + ' ' + accountManager.account_manager_last_name);
        });
      }
      let accountManagerDisplay = accountManagerNames.join(',');
      return (
        <Layout className="contracts--card" margin={{ bottom: 2 }} key={contract_id}>
          <Layout display={Display.Flex} justifyContent={JustifyContent.Between} alignItems={AlignItems.End} margin={{ bottom: 0.5 }} fullWidth>
            <Button dropdown size={ButtonSize.Large} type={ButtonType.Text} onClick={() => setActiveContract(contract_id)}>
              <CoreText type={TextType.H4}>{contract_name}</CoreText>
            </Button>
            <ButtonIcon icon={SVGAsset.Edit} onClick={() => editContract(contract)} />
          </Layout>
          <StyledLayout className="contracts--card-wrapper table--wrapper" background={Background.Base} fontSize={FontSize.Size7}>
            <Table>
              <TableHeader>
                <TableRow>
                  {[
                    'Account Manager',
                    'Timeline',
                    'Type',
                    'Ad Share',
                    'T1',
                    'T2',
                    'T3',
                    'Bits',
                    'Subs',
                    'Ad Density',
                    'Exclusivity',
                    'Prerolls',
                    'Sponsorship'
                  ].map(tableHeader => (
                    <TableHeading key={tableHeader} label={tableHeader} />
                  ))}
                </TableRow>
              </TableHeader>
              <TableBody>
                <TableRow className="contracts--row" key={contract_id} height={60}>
                  <TableCell width={230}>{accountManagerDisplay}</TableCell>
                  <TableCell width={180}>
                    {getMomentDate(start_date)} - {getMomentDate(end_date)}
                  </TableCell>
                  <TableCell width={170}>{contract_type}</TableCell>
                  <TableCell width={120}>{ad_revenue_contract_type}</TableCell>
                  <TableCell width={100}>
                    {ad_revenue_share_amount_tier_1}
                    {ad_revenue_share_type_tier_1 === 'cpm' ? ' CPM' : '%'}
                  </TableCell>
                  <TableCell width={100}>
                    {ad_revenue_share_amount_tier_2}
                    {ad_revenue_share_type_tier_2 === 'cpm' ? ' CPM' : '%'}
                  </TableCell>
                  <TableCell width={100}>
                    {ad_revenue_share_amount_tier_3}
                    {ad_revenue_share_type_tier_3 === 'cpm' ? ' CPM' : '%'}
                  </TableCell>
                  <TableCell width={100}>{bits_revenue_share_percent}%</TableCell>
                  <TableCell width={100}>{subs_revenue_share_percent}%</TableCell>
                  <TableCell width={100}>{ad_density_per_hour} min per hr</TableCell>
                  <TableCell width={100}>{exclusivity_enabled && <SVG asset={SVGAsset.Check} type={SVGType.Success} />}</TableCell>
                  <TableCell width={100}>{ad_prerolls_enabled && <SVG asset={SVGAsset.Check} type={SVGType.Success} />}</TableCell>
                  <TableCell width={100}>{sponsorship_enabled && <SVG asset={SVGAsset.Check} type={SVGType.Success} />}</TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </StyledLayout>
          {channels && channels.length && (
            <Layout margin={{ top: 2 }} display={openContract === contract_id ? Display.Block : Display.Hide}>
              <Transition type={TransitionType.SlideOverTop} show={openContract === contract_id}>
                <Layout margin={{ bottom: 1 }}>
                  <CoreText type={TextType.H5} fontWeight={FontWeight.Bold}>
                    Associated Channels &amp; Custom Dates (if applicable)
                  </CoreText>
                </Layout>
                <SimpleBarReact style={{ maxHeight: 190 }}>
                  <Layout display={Display.Flex} flexWrap={FlexWrap.Wrap}>
                    {channels &&
                      channels.map((channel, i) => {
                        const { channel_login, profile_image, custom_start_date, custom_end_date } = channel;
                        return (
                          <Layout
                            className="contracts-list--channel"
                            key={`${i}-${channel_login}`}
                            display={Display.Flex}
                            alignItems={AlignItems.Center}
                            margin={{ right: 1, bottom: 0.5 }}>
                            <Layout margin={0.5}>
                              <AvatarFallback login={null} profileImage={profile_image} size={40} />
                            </Layout>
                            <Layout>
                              <Layout display={Display.Block}>
                                <CoreText fontSize={FontSize.Size6} fontWeight={FontWeight.Bold} ellipsis>
                                  {channel_login || 'No Login found'}
                                </CoreText>
                              </Layout>
                              <Layout display={Display.Block}>
                                {custom_start_date != null && custom_end_date != null && (
                                  <CoreText fontSize={FontSize.Size6} color={Color.Alt}>
                                    {getMomentDate(custom_start_date)} - {getMomentDate(custom_end_date)}
                                  </CoreText>
                                )}
                              </Layout>
                            </Layout>
                          </Layout>
                        );
                      })}
                  </Layout>
                </SimpleBarReact>
              </Transition>
            </Layout>
          )}
          {additional_notes && additional_notes.length && (
            <Layout margin={{ top: 2 }} display={openContract === contract_id ? Display.Block : Display.Hide}>
              <Transition type={TransitionType.SlideOverTop} show={openContract === contract_id}>
                <Layout margin={{ top: 2 }}>
                  <CoreText type={TextType.H5} fontWeight={FontWeight.Bold}>
                    Additional Notes
                  </CoreText>
                </Layout>
                <Layout margin={{ bottom: 1 }}>
                  <CoreText type={TextType.P}>{additional_notes}</CoreText>
                </Layout>
              </Transition>
            </Layout>
          )}
          {document_link && document_link.length && (
            <Layout margin={{ top: 2 }} display={openContract === contract_id ? Display.Block : Display.Hide}>
              <Transition type={TransitionType.SlideOverTop} show={openContract === contract_id}>
                <Layout margin={{ top: 2 }}>
                  <CoreText type={TextType.H5} fontWeight={FontWeight.Bold}>
                    Google Doc
                  </CoreText>
                  <CoreLink linkTo={document_link} targetBlank>
                    <CoreText type={TextType.Span}>More Info</CoreText>
                    <Layout display={Display.Inline} margin={{ left: 0.5 }}>
                      <SVG asset={SVGAsset.Externallink} />
                    </Layout>
                  </CoreLink>
                </Layout>
              </Transition>
            </Layout>
          )}
        </Layout>
      );
    })
  );
}

export default ContractsList;
