import React, { useState } from 'react';
import { useHistory } from 'react-router-dom';
import { Layout } from 'twitch-core-ui';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import PaginationWrapper from './pagination-wrapper';
import ContractsHeader from './contracts-header';
import ContractsList from './contracts-list';
import ContractsPlaceholder from './contracts-placeholder';
import MetricsConsole from './metrics-console';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST } from '../hooks/useFetch';
import { useLocalStorage } from '@rehooks/local-storage';
import useUpdateEffect from '../hooks/useUpdateEffect';
import './styles/Contracts.scss';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function Contracts() {
  const history = useHistory();
  const searchParams = new URLSearchParams(history.location.search);
  const [currentPage, setCurrentPage] = useState(parseInt(searchParams.get('page')) || 1);
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const [filters, setFilters] = useState([
    { 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator },
    { 'filter-type': 7, key: 'end_date', v1: new Date().toISOString() }
  ]);
  const [isLegacyVisible, setIsLegacyVisible] = useState(false);

  let options = {
    limit: 10,
    page: currentPage - 1,
    eventtype: 1,
    filters: filters,
    'req-cols': ['*'],
    sortdata: [{ ord: 0, col: 'contract_name', asc: true }]
  };
  const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-contract-list`, options);
  const contracts = data == null || data.data == null ? [] : data.data.items;
  const totalPages = data == null || data.data == null ? 0 : data.data.total_pages;
  const metrics = data == null || data.metrics == null ? [] : data.metrics;

  useUpdateEffect(() => {
    setFilters([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);
  }, [selectedContentCreator]);

  useUpdateEffect(() => {
    if (isLegacyVisible) {
      setFilters([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);
    } else {
      setFilters([
        { 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator },
        { 'filter-type': 7, key: 'end_date', v1: new Date().toISOString() }
      ]);
    } /* eslint-disable react-hooks/exhaustive-deps */
  }, [isLegacyVisible]);

  useUpdateEffect(() => {
    setCurrentPage(currentPage);
    history.push({
      search: currentPage > 1 ? `?page=${currentPage}` : ''
    });
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, [currentPage, history, filters]);

  const editContract = contract => {
    history.push(`/contracts/edit/${contract.contract_id}`);
  };

  return (
    <>
      <Layout className="contracts">
        <ContractsHeader isLegacyVisible={isLegacyVisible} setIsLegacyVisible={setIsLegacyVisible} />
        {error && <ErrorAlert error={error} />}
        {loading ? (
          <ContractsPlaceholder />
        ) : contracts && contracts.length === 0 ? (
          <NoResults results="contracts" />
        ) : (
          <Layout className="contracts--card-wrapper">
            <ContractsList contracts={contracts} editContract={editContract} />
            {totalPages > 1 && <PaginationWrapper currentPage={currentPage} totalPages={totalPages} setCurrentPage={setCurrentPage} />}
          </Layout>
        )}
      </Layout>
      <MetricsConsole metrics={metrics} loading={loading} name="Contracts" />
    </>
  );
}

export default Contracts;
