import useLocalStorage from "@rehooks/local-storage";
import * as React from "react";
import { Background, BorderRadius, Button, CoreText, Display, FlexDirection, FontSize, Layout, LoadingSpinner, Overflow, Pill, PillType, ProgressBar, StyledLayout, WhiteSpace } from "twitch-core-ui";
import { useDataContext } from "../context/data-context";
import { parseCSV } from "../utils/csvParse";
import { downloadTextFile } from "../utils/helpers";
import { fuzzySelect } from "../utils/fuzzySearch";
import { EventParsingError, EventParsingResults, processEventValues } from "../utils/processEventValues";
import { AtlasEvent } from "../utils/types";
import { ChannelsExpando } from "./channels-expando";
import { DataItemTable, DataItemTableProps } from "./data-item-table";
import { ModalOverlay } from "./modal-overlay";
import { allowedEventImportHeaders, EventImportData, requiredEventImportHeaders } from "../utils/importModels";
import { generateEventsCsvTemplate } from "../utils/csvTemplate";
import { FileButton } from "./file-button";
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';
import { useGQLContext } from "../context/gql-context";

interface Props {
    onFileUploaded?: (contents: string) => void;
    onDone?: (results: AtlasEvent[]) => void;
    onCancel?: () => void;
}

/**
 * Columns that will be shown in the demo tables, after parsing the
 * input CSV. Note that these names can differ from the allowedHeaders above.
 */
const tableHeaders: (keyof AtlasEvent)[] = [
    "event_name",
    "product_name",
    "start_time",
    "end_time",
    "topic",
    "format",
    "costreaming_settings",
    "game_name",
    "channels",
    "estimated_average_ccv",
    "distribution",
];




/**
 * Special parsing rules during CSV parsing; these will transform how data is
 * converted for the specified properties.
 */
const parseRules: Partial<Record<keyof EventImportData, (string) => any>> = {
    "distribution": (value) => {
        const parsed = fuzzySelect(value, ["yes", "no", "true", "false"]);
        if (parsed === "yes" || parsed === "true") {
            return "true";
        } else {
            return "false";
        }
    }
}

/**
 * Custom drawing rules for cells that match the provided property names.
 * These are used to make the column for channels have a unique look.
 * Other cells will simply convert the value to a string.
 */
const customCellDrawers: Partial<Record<keyof AtlasEvent, (Object) => JSX.Element | null>> = {
    "channels": (item: Object) => {
        if (item && Array.isArray(item)) {
            return <ChannelsExpando channels={item} /> 
        }

        return null;
    },
};

interface ProgressUpdateDetails {
    progress: number;
    status: string;
}

export const CSVFileUploadModal: React.FC<Props> = (props) => {
    const { onCancel, onDone } = props;
    const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
    const [csvParseResults, setCsvParseResults] = React.useState<EventParsingResults | undefined>();
    const [parsingError, setParsingError] = React.useState<string | null>(null);
    const { getChannels } = useGQLContext();
    const { allTopicsFormatsOptions, productsList, loading } = useDataContext();
    const [parsingStatus, setParsingStatus] = React.useState<ProgressUpdateDetails | null>(null)

    const readFile = async (file: File) => {
        setParsingStatus({progress: 0, status: "Reading file"});
        try {
            const text = await file.text();
            try {
                const values = parseCSV(text, { 
                    knownHeaders: allowedEventImportHeaders,
                    rules: parseRules,
                    requiredHeaders: requiredEventImportHeaders,
                }) as EventImportData[];

                const updateCallback = (progress: number, status: string) => {
                    setParsingStatus({progress, status});
                }

                const results = await processEventValues(values, {
                    productsList,
                    currentContentCreatorID: selectedContentCreator,
                    allTopicsFormatsOptions,
                    getChannels,
                }, updateCallback);

                for (const parsingError of results.errors) {
                    console.error("Error parsing item", parsingError);
                }

                setCsvParseResults(results);
                setParsingError(null);
            } catch (err) {
                console.error("Failed to parse AtlasEvents", err);
                setParsingError(err?.message ?? "Unknown error");           
            }
        } catch (err) {
            const msg = "Error reading csv file";
            console.error(msg, err);
            setParsingError(msg);
        }

        setParsingStatus(null);
    }

    const onChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const files = e?.target?.files;
        if (files && files.length > 0) {
            const file = files[0];
            readFile(file);
        }
    }

    const onDownloadTemplateClick = () => {
        const template = generateEventsCsvTemplate(productsList, allTopicsFormatsOptions);
        downloadTextFile(template, "template.csv");
    }

    const events = csvParseResults?.parsedEvents;
    const errors = csvParseResults?.errors;

    const tableProps = {
        headers: tableHeaders,
        keyProperty: "event_name",
        customCellDrawers
    };

    // Format this data for use in the table
    const successTableData: DataItemTableProps = events ? { 
        items: events.map((data) => { return { data, }; }),
        label: "Successfully Read Events",
        type: "success",
        ...tableProps,
    } : null;
    const failureTableData: DataItemTableProps = errors ? { 
        items: errors.map((err) => { return { data: err.event, problemFields: err.problemFields }; }),
        label: "Events with Errors",
        type: "failure",
        ...tableProps,
    } : null;

    return (
        <ModalOverlay
            onClickOut={onCancel}
            title="Select a CSV file for upload"
            primaryButtonProps={{
                disabled: !events,
                children: "Done",
                onClick: () => onDone?.(events ?? []),
            }}
            secondaryButtonProps={{
                children: "Cancel",
                onClick: () => onCancel?.(),
            }}
            size={undefined} // unrestricted
            wide
        >
            {   
                loading ?
                ( 
                    <LoadingPanel /> 
                ) : 
                parsingStatus ?
                (
                    <ParsingProgressBar status={parsingStatus.status} progress={parsingStatus.progress} />
                ) :
                (
                    <Layout margin={{y: 1}}>
                        <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
                            <Layout margin={{right: 1 }}>
                                <Button onClick={onDownloadTemplateClick}>Download Template</Button>
                            </Layout>
                            <FileButton onChange={onChange} accept=".csv" text="Upload CSV" />
                        </Layout>
                        <Layout margin={{top: 2}}>
                            <DataItemTable {...successTableData} />
                        </Layout>
                        {parsingError && (
                            <Layout margin={{y: 1}}>
                                <CoreText bold fontSize={FontSize.Size4}>Parsing Error</CoreText>
                                <StyledLayout border overflow={Overflow.Auto} borderRadius={BorderRadius.Medium} padding={1} margin={1}>
                                    <CoreText whiteSpace={WhiteSpace.PreWrap}>{parsingError}</CoreText>
                                </StyledLayout>        
                            </Layout>
                        )}

                        {!!errors?.length && 
                            <Layout>
                                <Layout margin={{y: 2}}>
                                    <CoreText bold fontSize={FontSize.Size4}>Errors</CoreText>
                                    <ErrorTable errors={errors} />
                                </Layout>
                                <DataItemTable {...failureTableData} />
                            </Layout>
                        }
                        
                    </Layout>
                )
            }
        </ModalOverlay>
    );
}

const ParsingProgressBar: React.FC<ProgressUpdateDetails> = (props) => {
    if (!props) {
        return null;
    }

    const value = Math.min(100, Math.max(0, props.progress));
    return (
        <Layout>
            <ProgressBar mask value={value}/>
            <CoreText>{props.status}</CoreText>
        </Layout>
    )
}

const LoadingPanel: React.FC = () => {
    return (
        <Layout>
            <CoreText>Processing...</CoreText>
            <LoadingSpinner />
        </Layout>
    )
} 

const ErrorTable: React.FC<{errors?: EventParsingError[]}> = (props) => {
    const { errors } = props;
    if (!errors?.length) {
        return null;
    }

    return (
        <StyledLayout border overflow={Overflow.Auto} borderRadius={BorderRadius.Medium} margin={1}>
            {errors.map((item, i) => {
                return (
                    <StyledLayout key={i} 
                        flexDirection={FlexDirection.Row}
                        display={Display.Flex}
                        background={i % 2 === 0 ? Background.Alt2 : Background.Base } 
                        padding={1}
                    >
                        <Layout margin={{right: 1}}>
                            <Pill label={item.event?.event_name || "unknown"} type={PillType.Warn} />
                        </Layout>
                        <Layout flexDirection={FlexDirection.Column}>
                            { item.messages.map((message, j) => {
                                return <CoreText key={j} whiteSpace={WhiteSpace.PreWrap}>• {message}</CoreText>
                            })}
                        </Layout>
                    </StyledLayout>
                )
            })}
        </StyledLayout>
    );
}

export default CSVFileUploadModal;
