import classnames from "classnames";
import * as React from "react";
import { BorderRadius, CoreText, FontSize, Layout, StyledLayout, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, WhiteSpace } from "twitch-core-ui";
import "./styles/DataItemTable.scss";

export interface DataItemTableProps {
    label?: string;
    type?: "success" | "failure";
    items?: {
        data: Object,
        problemFields?: string[],
    }[];
    headers: string[],
    keyProperty: string,
    customCellDrawers?: {
        [index: string]: (data: Object) => JSX.Element | null;
    }
}

export const DataItemTable: React.FC<DataItemTableProps> = (props) => {
    if (!props?.items?.length) {
        return null;
    }

    const backdropClass = classnames({
        "item-table__success": props.type === "success",
        "item-table__failure": props.type === "failure",
        "item-table": true,
    });

    return (
        <StyledLayout padding={1} border borderRadius={BorderRadius.Medium} className={backdropClass}>
            <Layout margin={{ bottom: 1 }}>
                <CoreText bold fontSize={FontSize.Size4}>{props.label ?? "Items"}</CoreText>
            </Layout>
            <Table alternateRows>
                <TableHeader>
                <TableRow>
                    {props.headers.map((header, index) => {
                        const className = index % 2 === 0 ? "item-table-header__even" : "item-table-header__odd";
                        return <TableHeading key={header} label={header} className={className}/>
                    })}
                </TableRow>
                </TableHeader>
                <TableBody>
                    { props.items.map((item) => {
                        const keyProp = item.data[props.keyProperty];
                        if (!keyProp) {
                            return null;
                        }
                        return (
                            <TableRow key={keyProp} className={"item-table-row"}>
                                { props.headers.map((property) => {
                                    let className = "";
                                    if (item.problemFields?.includes(property)) {
                                        className = "item-table-cell__error";
                                    }

                                    let element: JSX.Element = null;

                                    // try to apply custom drawing rule
                                    if (props?.customCellDrawers) {
                                        const rule = props?.customCellDrawers[property];
                                        if (rule) {
                                            element = rule(item.data[property]); 
                                        }
                                    }

                                    // apply default string drawer
                                    if (!element) {
                                        let value = item.data[property];
                                        if (value !== undefined && value !== null) {
                                            value = value.toString();
                                            element = <CoreText whiteSpace={WhiteSpace.NoWrap}>{value}</CoreText>;
                                        }
                                    }
                                    return (
                                        <TableCell key={`${keyProp}_${property}`} className={className}>
                                            {element}
                                        </TableCell>
                                    );
                                })}
                            </TableRow>
                        );
                    })}
                </TableBody>
            </Table>
        </StyledLayout>
    )
}

export default DataItemTable;
