import React from 'react';
import { Layout, Input, InputType, InputProps, SVGAsset } from 'twitch-core-ui';
import Pikaday from 'pikaday';
import './styles/DatePicker.scss';

interface Props {
  defaultDate: Date;
  onChange: (defaultDate) => void;
  minDate?: Date;
  maxDate?: Date;
  inputProps?: Partial<InputProps>;
  position?: string;
  name?: string;
  disabled?: boolean;
  required?: boolean;
}

export class DatePicker extends React.Component<Props, {}> {
  private textInput: HTMLInputElement;
  private _picker: Pikaday;
  private containerElement: HTMLDivElement;

  private setInputRef = (e: HTMLInputElement) => (this.textInput = e);
  private setContainerRef = (e: HTMLDivElement) => (this.containerElement = e);
  private setDate(newDate?: Date) {
    if (!this._picker) {
      return;
    }

    if (!newDate) {
      // Workaround for pikaday not clearing value when date set to falsey
      this.textInput.value = '';
    }
    this._picker.setDate(newDate, true); // 2nd param = don't call onSelect
  }

  private setDateRangeIfPresent(minDate?: Date, maxDate?: Date) {
    if (!this._picker) {
      return;
    }

    if (minDate) {
      this._picker.setMinDate(minDate);
    }
    if (maxDate) {
      this._picker.setMaxDate(maxDate);
    }
  }

  private dateDidChange(prevDate: Date | undefined, newDate: Date | undefined) {
    const prevTime = prevDate ? prevDate.getTime() : null;
    const newTime = newDate ? newDate.getTime() : null;
    return prevTime !== newTime;
  }

  public async componentDidMount() {
    this._picker = new Pikaday({
      field: this.textInput,
      onSelect: this.props.onChange,
      minDate: this.props.minDate,
      maxDate: this.props.maxDate,
      defaultDate: this.props.defaultDate,
      setDefaultDate: true,
      container: this.containerElement,
      theme: 'inline',
      format: 'MM/DD/YYYY',
      position: this.props.position || 'left'
    });
  }

  public componentDidUpdate(prevProps: Props) {
    try {
      if (this.dateDidChange(prevProps.minDate, this.props.minDate) || this.dateDidChange(prevProps.maxDate, this.props.maxDate)) {
        this.setDateRangeIfPresent(this.props.minDate, this.props.maxDate);
      }
      if (this.dateDidChange(prevProps.defaultDate, this.props.defaultDate)) {
        this.setDate(this.props.defaultDate);
      }
    } catch (err) {}
  }

  componentWillUnmount() {
    this._picker.destroy();
  }

  render() {
    const { position, inputProps, name, disabled, required } = this.props;
    return (
      <Layout className="date-picker" data-a-target="date-picker">
        <Input
          type={InputType.Text}
          name={name}
          refDelegate={this.setInputRef}
          {...inputProps}
          readOnly={!this._picker}
          placeholder="MM/DD/YY"
          iconRight
          icon={SVGAsset.DateStart}
          disabled={disabled}
          error={required ? inputProps && inputProps.value.length !== 10 : false}
          required={required}
        />
        <div className={`date-picker__container ${position === 'right' ? 'date-picker__container--right' : ''}`} ref={this.setContainerRef} />
      </Layout>
    );
  }
}
export default DatePicker;
