import React from "react"
import { AlignItems, Background, CoreText, Display, JustifyContent, Layout, ModalSize, Select, StyledLayout } from "twitch-core-ui"
import { channelTypeOptions } from "../utils/options"
import { Channel, WorkspaceEvent } from "../utils/types"
import AvatarFallback from "./avatar-fallback"
import { ModalOverlay } from "./modal-overlay"

interface Props {
    events: WorkspaceEvent[];
    onDone?: () => void;
    onCancel?: () => void;
}

export const EditEventChannelsModal: React.FC<Props> = (props) => {

    const channelMap: Record<string, Channel> = {};
    for (const event of props.events) {
        for (const channel of event.channels ?? []) {
            // map and de-dupe channels
            channelMap[channel.channel_login] = channel;
        }
    }

    const channelNames = Object.keys(channelMap);
    channelNames.sort();

    const onDone = () => {
        // When we click "done", we want to apply the changes to ALL
        // selected items with the same channels, since we de-duped
        // channels to avoid having duplicates.
        for (const event of props.events) {
            for (const channel of event.channels ?? []) {
                const updated = channelMap[channel.channel_login];
                channel.channel_type = updated.channel_type;
            }
        }

        props.onDone?.();
    }

    const handleTypeSelect = (value: string, name: string) => {
        if (channelMap[name]) {
            channelMap[name].channel_type = value;
        }
    }

    return (
        <ModalOverlay
            title="Edit Channels"
            primaryButtonProps={{
                children: "Done",
                onClick: onDone,
            }}
            secondaryButtonProps={{
                children: "Cancel",
                onClick: () => props?.onCancel(),
            }}
            size={ModalSize.Medium}        
        >
            {channelNames.map((name, i) => {
                const channel = channelMap[name];
                const background = i % 2 === 0 ? Background.Base : Background.Alt;
                return (
                    <StyledLayout key={name} 
                        display={Display.Flex} 
                        padding={1} 
                        alignItems={AlignItems.Center} 
                        background={background} 
                        justifyContent={JustifyContent.Between}
                    >
                        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                            <AvatarFallback login={name} profileImage={channel.profile_image} size={30} />
                            <Layout margin={{left: 1}}><CoreText>{name}</CoreText></Layout>
                        </Layout>
                        <div style={{ height: 30, width: 180 }}>
                          <Select defaultValue={(channel && channel.channel_type) || '0'} onChange={e => handleTypeSelect(e.currentTarget.value, name)}>
                            <option disabled value="">
                              Select Channel Type
                            </option>
                            {channelTypeOptions.map(type => {
                              return (
                                <option key={type} value={type}>
                                  {type}
                                </option>
                              );
                            })}
                          </Select>
                        </div>
                    </StyledLayout>
                )
            })}
        </ModalOverlay>
    );
}
