import './styles/Events.scss';

import React, { useState } from 'react';

import { useHistory } from 'react-router-dom';
import {
  AlignItems,
  Color,
  CoreText,
  Display,
  FontSize,
  FontWeight,
  InjectStyledLayout,
  Layout,
  Tag,
  TextType,
} from 'twitch-core-ui';

import { useLocalStorage } from '@rehooks/local-storage';

import { EventsProvider } from '../context/events-context';
import { useFetchPOST } from '../hooks/useFetch';
import useUpdateEffect from '../hooks/useUpdateEffect';
import { ATLAS_BASE_URL } from '../utils/config';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';
import {
  buildEventFilters,
  hyphenify,
} from '../utils/helpers';
import ErrorAlert from './error-alert';
import EventsHeader from './events-header';
import EventsList from './events-list';
import EventsPlaceholder from './events-placeholder';
import MetricsConsole from './metrics-console';
import NoResults from './no-results';
import PaginationWrapper from './pagination-wrapper';

function EventsCategory() {
  const history = useHistory();
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const searchParams = new URLSearchParams(history.location.search);
  const [currentPage, setCurrentPage] = useState(parseInt(searchParams.get('page')) || 1);
  const [filters, setFilters] = useState(buildEventFilters(searchParams, selectedContentCreator));

  let options = {
    limit: 5,
    page: currentPage - 1,
    eventtype: 1,
    filters: filters,
    'req-cols': ['*']
  };
  const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-category-event-list`, options);
  const events = data == null || data.data == null ? {} : data.data;
  const totalPages = data == null || data.data == null ? 0 : data.data.total_pages;
  const metrics = data == null || data.metrics == null ? [] : data.metrics;

  useUpdateEffect(() => {
    setFilters(buildEventFilters(searchParams, selectedContentCreator)); /* eslint-disable react-hooks/exhaustive-deps */
  }, [selectedContentCreator]);

  useUpdateEffect(() => {
    setCurrentPage(currentPage);
    let params = {};
    for (let param of searchParams as any) {
      if (param[1].length) {
        params[param[0]] = param[1];
      }
    }
    if (currentPage !== 1 && params['page'] === undefined) {
      params['page'] = currentPage;
    } else if (params['page'] !== undefined) {
      params['page'] = currentPage;
    }
    history.push({
      search: new URLSearchParams(params).toString()
    });
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, [currentPage, history, filters]);

  return (
    <>
      <Layout className="events">
        <EventsHeader activePage="Category" loading={loading} />
        {error && <ErrorAlert error={error} />}
        {loading ? (
          <EventsPlaceholder type="category" />
        ) : events && events.groupings && events.groupings.length === 0 ? (
          <NoResults results="Category Events" />
        ) : (
          events &&
          events.groupings &&
          events.groupings.map(data => {
            const { category, viewers, events, tags } = data;
            return (
              <Layout key={category}>
                <CategoryHeader category={category} viewers={viewers} tags={tags} />
                <Layout margin={{ bottom: 3 }}>
                  {events && events.length === 0 ? (
                    <NoResults results={`${category} events`} />
                  ) : (
                    <EventsProvider>
                      <EventsList events={events} grouping={hyphenify(category)} />
                    </EventsProvider>
                  )}
                </Layout>
              </Layout>
            );
          })
        )}
        {totalPages > 1 && <PaginationWrapper currentPage={currentPage} totalPages={totalPages} setCurrentPage={setCurrentPage} />}
      </Layout>
      <MetricsConsole metrics={metrics} loading={loading} name="Events Category" />
    </>
  );
}

function CategoryHeader({ category, viewers, tags }) {
  return (
    <Layout margin={{ bottom: 2 }}>
      <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ bottom: 2 }}>
        <Layout>
          <img alt={`${category} box art`} src={`https://static-cdn.jtvnw.net/ttv-boxart/${category}-60x80.jpg`} />
        </Layout>
        <Layout padding={{ left: 2 }}>
          <CoreText type={TextType.H4} fontWeight={FontWeight.SemiBold}>
            {category}
          </CoreText>
          <CoreText type={TextType.H5} color={Color.Alt2}>
            {viewers} viewers
          </CoreText>
          {tags?.length ? (
            <InjectStyledLayout display={Display.Flex} margin={{ top: 0.5 }} fontSize={FontSize.Size6}>
              <ul>
                {tags.map(label => (
                  <li key={label}>
                    <Layout margin={{ right: 0.5 }}>
                      <Tag label={label} onClick={e => e.preventDefault()} />
                    </Layout>
                  </li>
                ))}
              </ul>
            </InjectStyledLayout>
          ) : null}
        </Layout>
      </Layout>
    </Layout>
  );
}

export default EventsCategory;
