import React, { useState } from 'react';
import { useHistory } from 'react-router-dom';
import { Layout } from 'twitch-core-ui';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import PaginationWrapper from './pagination-wrapper';
import EventsList from './events-list';
import ChannelCard from './channel-card';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST } from '../hooks/useFetch';
import EventsHeader from './events-header';
import EventsPlaceholder from './events-placeholder';
import MetricsConsole from './metrics-console';
import { useLocalStorage } from '@rehooks/local-storage';
import { buildEventFilters } from '../utils/helpers';
import { EventsProvider } from '../context/events-context';
import useUpdateEffect from '../hooks/useUpdateEffect';
import './styles/Events.scss';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function EventsChannel() {
  const history = useHistory();
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const searchParams = new URLSearchParams(history.location.search);
  const [currentPage, setCurrentPage] = useState(parseInt(searchParams.get('page')) || 1);
  const [filters, setFilters] = useState(buildEventFilters(searchParams, selectedContentCreator));

  let options = {
    limit: 10,
    page: currentPage - 1,
    eventtype: 1,
    filters: filters,
    'req-cols': ['*']
  };
  const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-channel-event-list`, options);
  const events = data == null || data.data == null ? [] : data.data;
  const totalPages = data == null || data.data == null ? 0 : data.data.total_pages;
  const metrics = data == null || data.metrics == null ? [] : data.metrics;

  useUpdateEffect(() => {
    setFilters(buildEventFilters(searchParams, selectedContentCreator)); /* eslint-disable react-hooks/exhaustive-deps */
  }, [selectedContentCreator]);

  useUpdateEffect(() => {
    setCurrentPage(currentPage);
    let params = {};
    for (let param of searchParams as any) {
      if (param[1].length) {
        params[param[0]] = param[1];
      }
    }
    if (currentPage !== 1 && params['page'] === undefined) {
      params['page'] = currentPage;
    } else if (params['page'] !== undefined) {
      params['page'] = currentPage;
    }
    history.push({
      search: new URLSearchParams(params).toString()
    });
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, [currentPage, history, filters]);

  return (
    <>
      <Layout className="events">
        <EventsHeader activePage="Channel" loading={loading} />
        {error && <ErrorAlert error={error} />}
        {loading ? (
          <EventsPlaceholder type="channel" />
        ) : events && events.groupings && events.groupings.length === 0 ? (
          <NoResults results="Channel Events" />
        ) : (
          events &&
          events.groupings &&
          events.groupings.map(data => {
            const { channel_login, channel_status, viewers, events, channel_category, channel_profile_image, channel_id } = data;
            return (
              <Layout key={channel_id} margin={{ bottom: 2 }}>
                <Layout margin={{ bottom: 1 }}>
                  <div style={{ width: 240 }}>
                    <ChannelCard
                      login={channel_login}
                      status={channel_status}
                      viewers={viewers}
                      category={channel_category}
                      profileImage={channel_profile_image}
                    />
                  </div>
                </Layout>
                <Layout margin={{ bottom: 3 }}>
                  {events && events.length === 0 ? (
                    <NoResults results={`${channel_login} events`} />
                  ) : (
                    <EventsProvider>
                      <EventsList events={events} grouping={channel_id} />
                    </EventsProvider>
                  )}
                </Layout>
              </Layout>
            );
          })
        )}
        {totalPages > 1 && <PaginationWrapper currentPage={currentPage} totalPages={totalPages} setCurrentPage={setCurrentPage} />}
      </Layout>
      <MetricsConsole metrics={metrics} loading={loading} name="Events Channel" />
    </>
  );
}

export default EventsChannel;
