import React, { useState } from 'react';
import { useHistory } from 'react-router-dom';
import SimpleBarReact from 'simplebar-react';
import Cookies from 'js-cookie';
import { v4 as uuidv4 } from 'uuid';
import {
  StyledLayout,
  Background,
  Position,
  CoreText,
  TextType,
  Layout,
  Display,
  Button,
  ButtonType,
  FontWeight,
  TextTransform,
  JustifyContent,
  AlignItems,
  Color,
  SplitButton,
  CoreButtonDropdownType,
  DropDownMenu,
  DropDownMenuItem,
  BalloonDirection,
  BalloonSize,
  FlexWrap,
  Overflow,
  ZIndex,
  Modal,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalSize
} from 'twitch-core-ui';
import { ATLAS_BASE_URL } from '../utils/config';
import { fetchWithTimeout } from '../utils/helpers';
import { useSelectedEvents } from '../context/events-context';
import ClickOutDetector from './click-out-detector';
import AvatarFallback from './avatar-fallback';
import Overlay from './overlay';
import NoResults from './no-results';

function EventsDrawer() {
  const history = useHistory();
  const [selectedEvent, setSelectedEvents] = useSelectedEvents();
  const [isDeleteDropdownDisplayed, setIsDeleteDropdownDisplayed] = useState(false);
  const [isDeleteModalDisplayed, setIsDeleteModalDisplayed] = useState(false);

  const deleteEvent = () => {
    let obj = {
      ...selectedEvent[0],
      is_active: false
    };

    const body = Object.keys(obj)
      .filter(e => obj[e] !== null)
      .reduce((o, e) => {
        o[e] = obj[e];
        return o;
      }, {});

    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(body)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-event`, options);
        const json = await response.json();
        console.log(`/edit-event:`, json);
        document.location.reload();
      } catch (error) {
        console.error(`/edit-event:`, error);
        // TODO: on error display error message somewhere
      }
    }
    fetchData();
    setIsDeleteModalDisplayed(false);
  };

  const addCopyToWorkspace = () => {
    // if eventsWorkspaceList localStorage
    let eventsWorkspaceArr = [];
    const eventsWorkspaceList = localStorage.getItem('eventsWorkspaceList');
    if (eventsWorkspaceList != null) {
      eventsWorkspaceArr = [...JSON.parse(eventsWorkspaceList)];
    }

    // create a copy
    const eventCopyObj = (({ event_id, ...o }) => o)(selectedEvent[0]);
    eventCopyObj['event_name'] += ` COPY ${!!eventsWorkspaceArr ? `(${eventsWorkspaceArr.length + 1})` : ''}`;
    eventCopyObj['workspace_id'] = uuidv4();
    eventsWorkspaceArr.push(eventCopyObj);

    // save to localStorage
    localStorage.setItem('eventsWorkspaceList', JSON.stringify(eventsWorkspaceArr));

    // go to events/workspace
    history.push('/events/workspace');
  };

  const currentEvent = selectedEvent[0];
  let channels = [];
  if (currentEvent && currentEvent.channels) {
    channels = [...selectedEvent[0].channels];
    if (channels.length) {
      channels.sort((a, b) => (a.channel_login > b.channel_login) ? 1 : -1)
    }
  }

  return selectedEvent && selectedEvent.length ? (
    <StyledLayout
      className="events--drawer"
      background={Background.Base}
      padding={{ y: 3, x: 5 }}
      elevation={4}
      position={Position.Fixed}
      attachBottom
      attachRight
      zIndex={ZIndex.Above}>
      <Layout className="drawer--header" margin={{ bottom: 3 }} display={Display.Flex} justifyContent={JustifyContent.Between}>
        <CoreText type={TextType.H3}>
          <strong>Additional Details</strong>: {selectedEvent[0].event_name}
        </CoreText>
        <Layout margin={{ left: 1 }} display={Display.Flex} alignItems={AlignItems.Start}>
          <Layout margin={{ left: 1 }}>
            <Button type={ButtonType.Secondary} onClick={() => setSelectedEvents([])}>
              Close
            </Button>
          </Layout>
          <Layout margin={{ left: 1 }}>
            <Button type={ButtonType.Secondary} onClick={addCopyToWorkspace}>
              Add Copy to Workspace
            </Button>
          </Layout>
          <Layout margin={{ left: 1 }} position={Position.Relative}>
            <SplitButton
              dropdown={{ type: CoreButtonDropdownType.ArrowDown, onClick: () => setIsDeleteDropdownDisplayed(true) }}
              linkTo={`/events/edit/${selectedEvent[0].event_id}`}>
              Edit Event
            </SplitButton>
            <ClickOutDetector onClickOut={() => setIsDeleteDropdownDisplayed(false)}>
              <DropDownMenu show={isDeleteDropdownDisplayed} direction={BalloonDirection.BottomRight} size={BalloonSize.Auto}>
                <DropDownMenuItem onClick={() => setIsDeleteModalDisplayed(true)}>Delete Event</DropDownMenuItem>
              </DropDownMenu>
            </ClickOutDetector>
            {isDeleteModalDisplayed && (
              <Overlay onClickOut={() => setIsDeleteModalDisplayed(false)}>
                <Modal size={ModalSize.Small}>
                  <ModalHeader
                    title="Delete Event"
                    closeButton={{
                      'aria-label': 'Close Modal',
                      onClick: () => setIsDeleteModalDisplayed(false)
                    }}
                  />
                  <ModalBody>
                    Are you sure you want to delete: <strong>"{selectedEvent[0].event_name}"</strong>
                  </ModalBody>
                  <ModalFooter
                    primaryButtonProps={{
                      children: 'Delete Event',
                      onClick: deleteEvent,
                      type: ButtonType.Alert
                    }}
                    secondaryButtonProps={{
                      children: 'Cancel',
                      onClick: e => {
                        e.preventDefault();
                        setIsDeleteModalDisplayed(false);
                      }
                    }}
                  />
                </Modal>
              </Overlay>
            )}
          </Layout>
        </Layout>
      </Layout>
      <Layout className="drawer--body" margin={{ bottom: 3 }} display={Display.Flex} flexWrap={FlexWrap.Wrap}>
        <DrawerItem drawerLabel="Season" drawerVal={selectedEvent[0].season_name} />
        <DrawerItem drawerLabel="Sub-Format" drawerVal={selectedEvent[0].sub_format} />
        <DrawerItem drawerLabel="Distribution Exclusivity" drawerVal={selectedEvent[0].distribution ? 'Yes' : 'No'} />
        <DrawerItem drawerLabel="Twitch Involvement" drawerVal={selectedEvent[0].twitch_involvement} />
        <DrawerItem drawerLabel="Co-Streaming Settings" drawerVal={selectedEvent[0].costreaming_settings} />
      </Layout>
      <Layout className="channels--wrapper" margin={{ bottom: 3 }}>
        <Layout margin={{ bottom: 1 }}>
          <CoreText fontWeight={FontWeight.SemiBold} transform={TextTransform.Uppercase}>
            Channels
          </CoreText>
        </Layout>
        {!!channels ? (
          <SimpleBarReact style={{ maxHeight: 160 }}>
            <Layout display={Display.Flex} flexWrap={FlexWrap.Wrap}>
              {channels.map(channel => {
                const { channel_id, channel_type, channel_login, profile_image } = channel;
                return (
                  <div key={channel_id} style={{ width: 190 }}>
                    <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ right: 1, bottom: 1 }}>
                      <Layout flexShrink={0}>
                        <AvatarFallback login={channel_login} profileImage={profile_image} size={30} />
                      </Layout>
                      <Layout margin={{ left: 0.5 }} overflow={Overflow.Hidden}>
                        <CoreText ellipsis>{channel_login}</CoreText>
                        {channel_type !== 'Unknown' && <CoreText color={Color.Alt2}>{channel_type}</CoreText>}
                      </Layout>
                    </Layout>
                  </div>
                );
              })}
            </Layout>
          </SimpleBarReact>
        ) : (
          <NoResults results="channels" />
        )}
      </Layout>
    </StyledLayout>
  ) : null;
}

export default EventsDrawer;

function DrawerItem({ drawerLabel, drawerVal }) {
  return (
    <div style={{ width: 200 }}>
      <Layout margin={{ bottom: 1 }}>
        <CoreText fontWeight={FontWeight.SemiBold} transform={TextTransform.Uppercase}>
          {drawerLabel}
        </CoreText>
      </Layout>
      <CoreText>{drawerVal}</CoreText>
    </div>
  );
}
