import React, { useState, useRef } from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  CoreText,
  Layout,
  Display,
  AlignItems,
  SearchInput,
  Position,
  DropDownMenu,
  DropDownMenuItem,
  Transition,
  TransitionType,
  ZIndex,
  BalloonSize,
  CoreImage,
  InjectStyledLayout,
  BorderRadius
} from 'twitch-core-ui';
import ClickOutDetector from './click-out-detector';
import Loading from './loading';
import NoResults from './no-results';
import useUpdateEffect from '../hooks/useUpdateEffect';
import './styles/Form.scss';
import { useGQLContext } from '../context/gql-context';
import { Category } from '../utils/gqlTypes';

function EventsDropdownGame({ game, setGame }) {
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);

  const { searchCategories } = useGQLContext(); 

  const [loading, setLoading] = useState(false);
  const [categories, setCategories] = useState<Category[] | null>([]);

  const timeoutRef = useRef(null);

  useUpdateEffect(() => {
    if (timeoutRef.current !== null) {
      clearTimeout(timeoutRef.current);
    }
    timeoutRef.current = setTimeout(async () => {
      timeoutRef.current = null;
      try {
        setLoading(true);
        const result = await searchCategories(game);
        setCategories(result.categories);
      } catch (err) {
        console.log("An error ocurred when searching for categories", err);
      }

      setLoading(false);
    }, 300);
    return () => {
      timeoutRef.current = false;
    }; /* eslint-disable react-hooks/exhaustive-deps */
  }, [game]);

  return (
    <ClickOutDetector onClickOut={() => setIsDropdownOpen(false)}>
      <SearchInput placeholder="Select Category" onFocus={() => setIsDropdownOpen(!isDropdownOpen)} onChange={e => setGame(e.currentTarget.value)} value={game} />
      <Layout position={Position.Relative} fullWidth zIndex={ZIndex.Above}>
        <Transition type={TransitionType.SlideOverTop} show={isDropdownOpen}>
          <DropDownMenu show={isDropdownOpen} size={BalloonSize.Auto}>
            <SimpleBarReact style={{ maxHeight: 300, width: 470 }}>
              {/* {error && <ErrorAlert error={error} />} */}
              {loading ? (
                <Loading />
              ) : !categories?.length ? (
                <NoResults results="games" />
              ) : (
                <div style={{ width: '100%' }}>
                  {categories.map(category => {
                    if (!category?.name) {
                      return null;
                    }

                    const name = category.name;
                    return (
                      <DropDownMenuItem
                        key={name}
                        onClick={e => {
                          e.preventDefault();
                          setGame(name);
                          setIsDropdownOpen(false);
                        }}>
                        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                          {category.avatarURL && (
                            <InjectStyledLayout margin={{x: 1}} borderRadius={BorderRadius.Medium}>
                              <CoreImage src={category.avatarURL} alt={name}/>
                            </InjectStyledLayout>
                          )}
                          <CoreText>{name}</CoreText>
                        </Layout>
                      </DropDownMenuItem>
                    );
                  })}
                </div>
              )}
            </SimpleBarReact>
          </DropDownMenu>
        </Transition>
      </Layout>
    </ClickOutDetector>
  );
}

export default EventsDropdownGame;
