import React, { useState, useEffect } from 'react';
import { useHistory, useParams } from 'react-router-dom';
import Cookies from 'js-cookie';
import {
  Layout,
  CoreText,
  TextType,
  StyledLayout,
  FormGroup,
  FormGroupOrientation,
  Input,
  InputType,
  Select,
  Button,
  ButtonType,
  Display,
  JustifyContent,
  ButtonState,
  Background,
  BorderRadius,
  AlignItems,
  CheckBox
} from 'twitch-core-ui';
import EventsDropdownGame from './events-dropdown-game';
import FormInputTime from './form-input-time';
import FormChannelsSelect from './form-channels-select';
import DatePicker from './date-picker';
import Loading from './loading';
import ErrorAlert from './error-alert';
import { ATLAS_BASE_URL } from '../utils/config';
import { channelTypeOptions, coStreamingSettingsOptions, timezoneOptions } from '../utils/options';
import { useFetchPOST } from '../hooks/useFetch';
import { useLocalStorage } from '@rehooks/local-storage';
import { fetchWithTimeout } from '../utils/helpers';
import { getDefaultTimezoneOption, createDateString, getMomentDate, getMomentTime } from '../utils/dateTimeHelpers';
import { useDataContext } from '../context/data-context';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';
import './styles/Form.scss';
import { Channel } from '../utils/types';


function EventsEdit() {
  const history = useHistory();
  const [isFormSubmitting, setIsFormSubmitting] = useState(false);
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const [error, setError] = useState('');

  let { id } = useParams<{ id: string }>();
  let options = {
    limit: 1,
    page: 0,
    eventtype: 1,
    'req-cols': ['*'],
    filters: [{ 'filter-type': 1, key: 'event_id', v1: id }]
  };
  const [data, loadingEvents] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-event-list`, options);

  // product, topics, & formats premium-content-creator options
  const { loading: loadingData, productsList, allTopicsFormatsOptions } = useDataContext();
  const [formatOptions, setFormatOptions] = useState([]);
  // page 2
  const [exclusivity, setExclusivity] = useState(false);
  // page 3
  const [timezone, setTimezone] = useState(
    localStorage.getItem('timezone') === null ? getDefaultTimezoneOption().value : JSON.parse(localStorage.getItem('timezone')).value
  );
  // page 4
  const [coStreamingSettings, setCoStreamingSettings] = useState(coStreamingSettingsOptions[0]);
  // page 5
  const [channels, setChannels] = useState<Channel[]>([]);

  // event data to send to backend
  const [eventId, setEventId] = useState(null);
  const [productId, setProductId] = useState('');
  const [eventName, setEventName] = useState('');
  const [startDate, setStartDate] = useState(null);
  const [startTime, setStartTime] = useState('');
  const [isInvalidStartTime, setIsInvalidStartTime] = useState(false);
  const [endDate, setEndDate] = useState(null);
  const [endTime, setEndTime] = useState('');
  const [isInvalidEndTime, setIsInvalidEndTime] = useState(false);
  const [topic, setTopic] = useState('');
  const [format, setFormat] = useState('');
  const [estimatedAverageCCV, setEstimatedAverageCCV] = useState('');
  const [category, setCategory] = useState('');

  useEffect(() => {
    if (data !== undefined && data.data !== undefined) {
      const {
        event_id,
        product_id,
        event_name,
        start_time,
        end_time,
        topic,
        format,
        estimated_average_ccv,
        game_name,
        distribution,
        costreaming_settings,
        channels
      } = data.data.items[0];

      setEventId(event_id || '0');
      setProductId(product_id || '');
      if (product_id) {
        handleProductChange(product_id);
      }
      setEventName(event_name || '');
      setStartDate(start_time ? new Date(getMomentDate(start_time)) : null);
      setStartTime(start_time ? getMomentTime(start_time) : '');
      setEndDate(end_time ? new Date(getMomentDate(end_time)) : null);
      setEndTime(end_time ? getMomentTime(end_time) : '');
      setTopic(topic || '');
      setFormat(format || '');
      setEstimatedAverageCCV(estimated_average_ccv || '0');
      setCategory(game_name || '');
      // For some unknown reason, Resonance expects a boolean value inside a string here
      setExclusivity(!!distribution && distribution === 'true' ? true : false);
      setCoStreamingSettings(costreaming_settings || 'Not allowed');
      setChannels(channels || []);
    } /* eslint-disable react-hooks/exhaustive-deps */
  }, [data]);

  const submitForm = () => {
    let eventObj = {
      event_id: eventId,
      product_id: productId === '' ? 0 : productId,
      event_name: eventName,
      premium_content_creator_id: selectedContentCreator,
      start_time: createDateString(startDate.getFullYear(), startDate.getMonth() + 1, startDate.getDate()) + 'T' + startTime + ':00',
      end_time: createDateString(endDate.getFullYear(), endDate.getMonth() + 1, endDate.getDate()) + 'T' + endTime + ':00',
      topic: topic,
      format: format,
      event_type: 'Demo',
      estimated_average_ccv: estimatedAverageCCV === '' || estimatedAverageCCV == null ? 0 : parseInt(estimatedAverageCCV),
      is_active: true,
      costreaming_settings: coStreamingSettings,
      game_name: category,
      // For some unknown reason, Resonance expects a boolean value inside a string here, so we can't return a boolean
      distribution: exclusivity ? 'true' : 'false',
      // The following fields are deprecated, but Resonance will reject the request if they are null:
      pledged_hours_broadcast: 0,
      season_id: 0,
      sub_format: '',
      sub_topic: '',
      twitch_involvement: ''
    };

    // ensure channels have a channel_type and event_id associated
    const eventChannels = channels.map((channel) => {
      return {
        ...channel,
        event_id: eventId,
        channel_type: channel.channel_type ?? channelTypeOptions[0],
      };
    });

    let obj = {
      event: eventObj,
      streams: eventChannels,
      time_zone: timezone
    };

    setIsFormSubmitting(true);
    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(obj)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-event-with-streams`, options);
        const json = await response.json();
        console.log(`/edit-event:`, json);

        if (json.success) {
          history.push('/events');
        } else {
          setError(!!json.error_message ? json.error_message : 'Edit Event request not successful');
          setIsFormSubmitting(false);
        }
      } catch (err) {
        console.error(`/edit-event:`, err);
        setError(err);
        setIsFormSubmitting(false);
      }
    }
    fetchData();
  };

  const handleProductChange = (productId: string): void => {
    setProductId(productId);
  };

  const reloadFormatOptions = (productId: string) => {
    if (!productId || !productsList || !allTopicsFormatsOptions) {
      return;
    }

    // set topic and subtopic based on selected product
    const selectedProduct = productsList?.find(prod => prod.product_id === parseInt(productId));
    if (selectedProduct) {
      const { topic } = selectedProduct;
      setTopic(topic);
      if (topic && allTopicsFormatsOptions) {
        // create format options based on selected product
        const topicOptions = allTopicsFormatsOptions?.find(obj => obj.topic === topic);
        if (topicOptions?.formats?.length) {
          setFormatOptions(topicOptions.formats);
          const currentFormat = topicOptions.formats.find(a => a.format === format);
          if (currentFormat) {
            // configured format is found in list
            setFormat(currentFormat.format);
          } else {
            // configured format not found in list - default to first item
            setFormat(topicOptions.formats[0].format);
          }
        } else {
          // no format list found - set to empty
          setFormat('');
        }
      }
    }
  }

  React.useEffect(() => {
    if (productId && productsList && allTopicsFormatsOptions) {
      reloadFormatOptions(productId);
    }
  }, [allTopicsFormatsOptions, productId, productsList]);

  const handleFormatChange = (e: React.FormEvent<HTMLSelectElement>): void => {
    var formatID = e.currentTarget.value;
    setFormat(formatID);
  };

  const isFormValid =
    !!eventName &&
    !!productId &&
    !!category &&
    !!format &&
    !!coStreamingSettings &&
    !!startDate &&
    !!startTime &&
    !!endDate &&
    !!endTime &&
    !isInvalidStartTime &&
    !isInvalidEndTime &&
    !!timezone &&
    !!channels &&
    channels.length;

  return (
    <Layout className="form--wrapper" margin="auto">
      <Layout padding={{ bottom: 1 }}>
        <CoreText type={TextType.H3}>Edit Event</CoreText>
      </Layout>

      {loadingEvents || loadingData ? (
        <Loading />
      ) : (
        <>
          <Layout margin={{ bottom: 2 }}>
            <CoreText type={TextType.H4}>Event Details</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Name" hint="Name your event" required>
                <Input type={InputType.Text} placeholder="Event Name" value={eventName} onChange={e => setEventName(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Product" hint="The product under which this event falls" required>
                <Select value={productId} onChange={e => handleProductChange(e.currentTarget.value)}>
                  <option disabled value="">
                    Select Product
                  </option>
                  {productsList?.map(option => (
                      <option value={option.product_id} key={option.product_id}>
                        {option.product_name}
                      </option>
                    ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Category" hint="Category the event is broadcasted under" required>
                <EventsDropdownGame game={category} setGame={setCategory} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Format" hint="The format of your event" required>
                <Select value={format} onChange={handleFormatChange} disabled={format === '0'}>
                  <option disabled value="">
                    Select Format
                  </option>
                  {formatOptions?.map(option => (
                      <option value={option.format} key={`format-${option.format}`}>
                        {option.format}
                      </option>
                    ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Co-streaming" hint="Is co-streaming allowed on this stream? Defaults to ‘Not allowed’" required>
                <Select value={coStreamingSettings} onChange={e => setCoStreamingSettings(e.currentTarget.value)}>
                  {coStreamingSettingsOptions.map(option => (
                    <option key={option} value={option}>
                      {option}
                    </option>
                  ))}
                </Select>
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          <Layout margin={{ bottom: 2, top: 5 }}>
            <CoreText type={TextType.H3}>Date &amp; Time</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Timezone" hint="The timezone being used for this event’s scheduling" required>
                <Select value={timezone} onChange={e => setTimezone(e.currentTarget.value)}>
                  <option disabled value="">
                    Select Timezone
                  </option>
                  {timezoneOptions.map(option => (
                    <option key={option.name} value={option.value}>
                      {option.name}
                    </option>
                  ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="Start Date &amp; Time"
                error={isInvalidStartTime}
                errorMessage="Sorry, your time is not valid"
                required>
                <Layout display={Display.Flex} alignItems={AlignItems.Start} fullWidth>
                  <Layout margin={{ right: 2 }} fullWidth>
                    <DatePicker defaultDate={startDate} onChange={setStartDate} maxDate={endDate} />
                  </Layout>
                  <Layout fullWidth>
                    <FormInputTime time={startTime} setTime={setStartTime} isInvalidTime={isInvalidStartTime} setIsInvalidTime={setIsInvalidStartTime} />
                  </Layout>
                </Layout>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="End Date &amp; Time"
                error={isInvalidEndTime}
                errorMessage="Sorry, your time is not valid"
                required>
                <Layout display={Display.Flex} alignItems={AlignItems.Start} fullWidth>
                  <Layout margin={{ right: 2 }} fullWidth>
                    <DatePicker defaultDate={endDate} onChange={setEndDate} minDate={startDate} />
                  </Layout>
                  <Layout fullWidth>
                    <FormInputTime time={endTime} setTime={setEndTime} isInvalidTime={isInvalidEndTime} setIsInvalidTime={setIsInvalidEndTime} />
                  </Layout>
                </Layout>
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          <Layout margin={{ bottom: 2, top: 5 }}>
            <CoreText type={TextType.H3}>Channel Management</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormChannelsSelect channels={channels} setChannels={setChannels} channelType="type" />
            </StyledLayout>
          </StyledLayout>

          <Layout margin={{ bottom: 2, top: 5 }}>
            <CoreText type={TextType.H3}>Other Details</CoreText>
          </Layout>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
            <FormGroup orientation={FormGroupOrientation.Horizontal} label="Distribution Exclusivity" hint="Does Twitch have the exclusive right to broadcast this content?">
              <CheckBox
                label="Exclusive to Twitch"
                onChange={e => setExclusivity(e.currentTarget.checked)}/>
            </FormGroup>
          </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Estimated Average CCV" hint="The estimated average CCV of your event">
                <Input
                  type={InputType.Number}
                  placeholder="Estimated Average CCV"
                  value={estimatedAverageCCV}
                  onChange={e => setEstimatedAverageCCV(e.currentTarget.value)}
                />
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          {error && (
            <Layout margin={{ y: 3 }}>
              <ErrorAlert error={error} />
            </Layout>
          )}

          <Layout margin={{ y: 3 }} display={Display.Flex} justifyContent={JustifyContent.End}>
            <Layout margin={{ right: 1 }}>
              <Button
                type={ButtonType.Secondary}
                linkTo="/events"
                state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default}
                disabled={isFormSubmitting}>
                Cancel
              </Button>
            </Layout>
            <Button onClick={submitForm} state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default} disabled={isFormSubmitting || !isFormValid}>
              Submit
            </Button>
          </Layout>
        </>
      )}
    </Layout>
  );
}

export default EventsEdit;
