import React, { useState, useEffect, useRef } from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  CoreText,
  Layout,
  Display,
  AlignItems,
  SearchInput,
  Position,
  DropDownMenu,
  DropDownMenuItem,
  Transition,
  TransitionType,
  ZIndex,
  Button,
  ButtonType,
  JustifyContent,
  Input,
  InputType,
  Pill,
  PillType,
  FlexWrap,
  Tag,
  TagAction,
  FontWeight,
  BalloonSize
} from 'twitch-core-ui';
import ClickOutDetector from './click-out-detector';
import Loading from './loading';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST } from '../hooks/useFetch';
import './styles/Form.scss';
import AvatarFallback from './avatar-fallback';

function EventsFilterDropdownChannel({ channels, setChannels }) {
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedOptions, setSelectedOptions] = useState(channels || []);
  const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/user-auto-complete`, { search: searchQuery });
  const userOptions = data == null || data.data == null ? [] : data.data;
  const timeoutRef = useRef(null);

  useEffect(() => {
    if (timeoutRef.current !== null) {
      clearTimeout(timeoutRef.current);
    }
    timeoutRef.current = setTimeout(() => {
      timeoutRef.current = null;
      fetchData();
    }, 300);
    return () => {
      timeoutRef.current = false;
    }; /* eslint-disable react-hooks/exhaustive-deps */
  }, [searchQuery]);

  const toggleDropdown = e => {
    e.preventDefault();
    setIsDropdownOpen(!isDropdownOpen);
  };

  const selectOption = (e, option) => {
    e.preventDefault();
    localStorage.setItem('eventsFilterChannels', JSON.stringify([...selectedOptions, option]));
    setSelectedOptions([...selectedOptions, option]);
  };

  const clearFilter = e => {
    e.preventDefault();
    setIsDropdownOpen(false);
    setSelectedOptions([]);
    localStorage.removeItem('eventsFilterChannels');
    setSelectedOptions([]);
  };

  const submitFilter = e => {
    e.preventDefault();
    setIsDropdownOpen(false);
    let channelStr = selectedOptions.map(option => option.channel_id).toString();
    setChannels(channelStr);
  };

  const removeSelectedOption = (e, option) => {
    e.preventDefault();
    localStorage.setItem('eventsFilterChannels', JSON.stringify([...selectedOptions.filter(o => o !== option)]));
    setSelectedOptions(selectedOptions.filter(o => o !== option));
  };

  return (
    <>
      <Layout margin={{ bottom: 0.5 }}>
        <CoreText bold>Channels</CoreText>
      </Layout>
      <ClickOutDetector onClickOut={() => setIsDropdownOpen(false)}>
        <Button type={ButtonType.Secondary} fullWidth dropdown onClick={toggleDropdown}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Layout margin={{ right: 0.5 }}>
              <CoreText fontWeight={FontWeight.Regular}>Select Channels</CoreText>
            </Layout>
            <Layout display={selectedOptions && selectedOptions.length ? Display.Block : Display.Hide}>
              <Pill type={PillType.Brand} label={selectedOptions && selectedOptions.length.toString()} />
            </Layout>
          </Layout>
        </Button>
        <Layout display={Display.Hide}>
          <Input
            type={InputType.Text}
            name="channels"
            value={selectedOptions && selectedOptions.map(option => option.channel_id).toString()}
            onChange={e => setChannels(e.currentTarget.value)}
          />
        </Layout>
        <Layout className="events-filter--dropdown" position={Position.Relative} fullWidth zIndex={ZIndex.Above}>
          <Transition type={TransitionType.SlideOverTop} show={isDropdownOpen}>
            <DropDownMenu show={isDropdownOpen} size={BalloonSize.Medium}>
              <Layout margin={{ bottom: 0.5 }}>
                <SearchInput placeholder="Search Channel(s)" onChange={e => setSearchQuery(e.currentTarget.value)} />
              </Layout>
              <SimpleBarReact style={{ maxHeight: 225 }}>
                <Layout display={selectedOptions && selectedOptions.length ? Display.Flex : Display.Hide} flexWrap={FlexWrap.Wrap}>
                  {selectedOptions &&
                    selectedOptions.map((option, i) => (
                      <Layout key={i} margin={{ bottom: 0.5, right: 0.5 }} ellipsis>
                        <Tag action={TagAction.Remove} label={option.channel_login} onClick={e => removeSelectedOption(e, option)} />
                      </Layout>
                    ))}
                </Layout>
                {error && <ErrorAlert error={error} />}
                {loading ? (
                  <Loading />
                ) : userOptions.length ? (
                  userOptions
                    .filter(
                      channelOption => selectedOptions && selectedOptions.every(selectedChannels => selectedChannels.channel_id !== channelOption.channel_id)
                    )
                    .map(channel => {
                      const { channel_id, channel_login, profile_image } = channel;
                      return (
                        <DropDownMenuItem key={channel_id} onClick={e => selectOption(e, channel)}>
                          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                            <Layout margin={{ right: 1 }}>
                              <AvatarFallback profileImage={profile_image} login={channel_login} size={24} />
                            </Layout>
                            <CoreText>{channel_login}</CoreText>
                          </Layout>
                        </DropDownMenuItem>
                      );
                    })
                ) : (
                  <NoResults results="channels" small />
                )}
              </SimpleBarReact>
              <Layout margin={{ top: 1 }} display={Display.Flex} justifyContent={JustifyContent.End}>
                <Layout margin={{ right: 1 }}>
                  <Button type={ButtonType.Text} onClick={clearFilter}>
                    Clear
                  </Button>
                </Layout>
                <Button onClick={submitFilter}>Submit</Button>
              </Layout>
            </DropDownMenu>
          </Transition>
        </Layout>
      </ClickOutDetector>
    </>
  );
}

export default EventsFilterDropdownChannel;
