import './styles/Form.scss';

import React, {
  useEffect,
  useRef,
  useState,
} from 'react';

import SimpleBarReact from 'simplebar-react';
import {
  AlignItems,
  BalloonSize,
  Button,
  ButtonType,
  CoreText,
  Display,
  DropDownMenu,
  DropDownMenuItem,
  FlexWrap,
  FontWeight,
  Input,
  InputType,
  JustifyContent,
  Layout,
  Pill,
  PillType,
  Position,
  SearchInput,
  Tag,
  TagAction,
  Transition,
  TransitionType,
  ZIndex,
} from 'twitch-core-ui';

import { useGQLContext } from '../context/gql-context';
import ClickOutDetector from './click-out-detector';
import Loading from './loading';
import NoResults from './no-results';

function EventsFilterDropdownGame({ games, setGames }) {
  const { searchCategories } = useGQLContext(); 
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedOptions, setSelectedOptions] = useState(games.length ? games.split(',') : []);
  const [loading, setLoading] = useState(false);
  const [gameOptions, setGameOptions] = useState<string[] | null>(null);
  // const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/game-auto-complete`, { search: searchQuery });
  // const gameOptions = data == null || data.data == null ? [] : data.data;

  const timeoutRef = useRef(null);

  useEffect(() => {
    if (timeoutRef.current !== null) {
      clearTimeout(timeoutRef.current);
    }
    timeoutRef.current = setTimeout(async () => {
      timeoutRef.current = null;
      try {
        setLoading(true);
        const result = await searchCategories(searchQuery);
        setGameOptions(result.categories.map(category => category.name));
      } catch (err) {
        console.log("An error ocurred when searching for categories", err);
      }

      setLoading(false);
    }, 300);
    return () => {
      timeoutRef.current = false;
    }; /* eslint-disable react-hooks/exhaustive-deps */
  }, [searchQuery]);

  const toggleDropdown = e => {
    e.preventDefault();
    setIsDropdownOpen(!isDropdownOpen);
  };

  const selectOption = (e, option) => {
    e.preventDefault();
    setSelectedOptions([...selectedOptions, option]);
  };

  const clearFilter = e => {
    e.preventDefault();
    setIsDropdownOpen(false);
    setSelectedOptions([]);
  };

  const submitFilter = e => {
    e.preventDefault();
    setIsDropdownOpen(false);
    let gamesStr = selectedOptions.map(option => option.channel_login).toString();
    setGames(gamesStr);
  };

  const removeSelectedOption = (e, option) => {
    e.preventDefault();
    setSelectedOptions(selectedOptions.filter(o => o !== option));
  };

  return (
    <>
      <Layout margin={{ bottom: 0.5 }}>
        <CoreText bold>Categories</CoreText>
      </Layout>
      <ClickOutDetector onClickOut={() => setIsDropdownOpen(false)}>
        <Button type={ButtonType.Secondary} dropdown fullWidth onClick={toggleDropdown}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Layout margin={{ right: 0.5 }}>
              <CoreText fontWeight={FontWeight.Regular}>Select Categories</CoreText>
            </Layout>
            <Layout display={selectedOptions && selectedOptions.length ? Display.Block : Display.Hide}>
              <Pill type={PillType.Brand} label={selectedOptions && selectedOptions.length.toString()} />
            </Layout>
          </Layout>
        </Button>
        <Layout display={Display.Hide}>
          <Input
            type={InputType.Text}
            name="gameName"
            value={selectedOptions && selectedOptions.map(game => game).toString()}
            onChange={e => setGames(e.currentTarget.value)}
          />
        </Layout>
        <Layout className="events-filter--dropdown" position={Position.Relative} fullWidth zIndex={ZIndex.Above}>
          <Transition type={TransitionType.SlideOverTop} show={isDropdownOpen}>
            <DropDownMenu show={isDropdownOpen} size={BalloonSize.Medium}>
              <Layout margin={{ top: 0.5, bottom: 0.5 }}>
                <SearchInput placeholder="Category" onChange={e => setSearchQuery(e.currentTarget.value)} />
              </Layout>
              <SimpleBarReact style={{ maxHeight: 225 }}>
                <Layout
                  display={selectedOptions && selectedOptions.length ? Display.Flex : Display.Hide}
                  flexWrap={FlexWrap.Wrap}
                  alignItems={AlignItems.Start}>
                  {selectedOptions &&
                    selectedOptions.map((game, i) => (
                      <Layout key={i} margin={{ bottom: 0.5, right: 0.5 }} ellipsis>
                        <Tag action={TagAction.Remove} label={game} onClick={e => removeSelectedOption(e, game)} />
                      </Layout>
                    ))}
                </Layout>
                {/* {error && <ErrorAlert error={error} />} */}
                {loading ? (
                  <Loading />
                ) : gameOptions?.length ? (
                  gameOptions
                    .filter(gameOption => selectedOptions && selectedOptions.every(selectedGames => selectedGames !== gameOption))
                    .map(game => {
                      return (
                        <DropDownMenuItem key={game} onClick={e => selectOption(e, game)}>
                          <CoreText>{game}</CoreText>
                        </DropDownMenuItem>
                      );
                    })
                ) : (
                  <NoResults results="categories" small />
                )}
              </SimpleBarReact>
              <Layout margin={{ top: 1 }} display={Display.Flex} justifyContent={JustifyContent.End}>
                <Layout margin={{ right: 1 }}>
                  <Button type={ButtonType.Text} onClick={clearFilter}>
                    Clear
                  </Button>
                </Layout>
                <Button onClick={submitFilter}>Confirm</Button>
              </Layout>
            </DropDownMenu>
          </Transition>
        </Layout>
      </ClickOutDetector>
    </>
  );
}

export default EventsFilterDropdownGame;
