import React, { useState } from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  CoreText,
  Layout,
  Display,
  AlignItems,
  Position,
  DropDownMenu,
  DropDownMenuItem,
  BalloonSize,
  Transition,
  TransitionType,
  ZIndex,
  Button,
  ButtonType,
  JustifyContent,
  Input,
  InputType,
  Pill,
  PillType,
  FlexWrap,
  Tag,
  TagAction,
  FontWeight,
  FontSize,
  Color
} from 'twitch-core-ui';
import ClickOutDetector from './click-out-detector';
import './styles/Form.scss';
import NoResults from './no-results';

function EventsFilterDropdownLocalStorage({
  query,
  setQuery,
  dropdownOptions,
  filterName = '',
  label = '',
  disabled = false,
  hint = '',
  localStorageKey = '',
  optionName = '',
  optionId = ''
}) {
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [selectedOptions, setSelectedOptions] = useState(query || []);

  const toggleDropdown = e => {
    e.preventDefault();
    setIsDropdownOpen(!isDropdownOpen);
  };

  const selectOption = (e, option) => {
    e.preventDefault();
    localStorage.setItem(localStorageKey, JSON.stringify([...selectedOptions, option]));
    setSelectedOptions([...selectedOptions, option]);
  };

  const clearFilter = e => {
    e.preventDefault();
    setIsDropdownOpen(false);
    setQuery('');
    setSelectedOptions([]);
    localStorage.removeItem(localStorageKey);
  };

  const submitFilter = e => {
    e.preventDefault();
    setIsDropdownOpen(false);
    const lsItem = localStorage.getItem(localStorageKey);
    setQuery(!!lsItem ? JSON.parse(localStorage.getItem(localStorageKey)) : '');
  };

  const removeSelectedOption = (e, option) => {
    e.preventDefault();
    setSelectedOptions(selectedOptions.filter(o => o !== option));
    localStorage.setItem(localStorageKey, JSON.stringify([...selectedOptions.filter(o => o !== option)]));
  };

  return (
    <>
      <Layout margin={{ bottom: 0.5 }}>
        <CoreText bold color={!!disabled ? Color.Alt2 : Color.Base}>
          {label}
        </CoreText>
      </Layout>
      <ClickOutDetector onClickOut={() => setIsDropdownOpen(false)}>
        <Button type={ButtonType.Secondary} dropdown fullWidth onClick={toggleDropdown} disabled={!!disabled}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Layout margin={{ right: 0.5 }}>
              <CoreText fontWeight={FontWeight.Regular} color={!!disabled ? Color.Alt2 : Color.Base}>
                Select {label}
              </CoreText>
            </Layout>
            <Layout display={selectedOptions?.length ? Display.Block : Display.Hide}>
              <Pill type={PillType.Brand} label={selectedOptions?.length.toString()} />
            </Layout>
          </Layout>
        </Button>
        {!!disabled && (
          <Layout margin={{ top: 0.5 }}>
            <CoreText fontSize={FontSize.Size7} color={Color.Alt2}>
              {hint}
            </CoreText>
          </Layout>
        )}
        <Layout display={Display.Hide}>
          <Input
            type={InputType.Text}
            name={filterName}
            value={selectedOptions?.length ? selectedOptions.map(option => option[optionId]).toString() : ''}
            onChange={e => setQuery(e.currentTarget.value)}
          />
        </Layout>
        <Layout className="events-filter--dropdown" position={Position.Relative} fullWidth zIndex={ZIndex.Above}>
          <Transition type={TransitionType.SlideOverTop} show={isDropdownOpen}>
            <DropDownMenu show={isDropdownOpen} size={BalloonSize.Medium}>
              <SimpleBarReact style={{ maxHeight: 300 }}>
                <Layout
                  display={selectedOptions?.length ? Display.Flex : Display.Hide}
                  flexWrap={FlexWrap.Wrap}
                  alignItems={AlignItems.Start}
                  fullWidth>
                  {selectedOptions?.length ? selectedOptions.map((option, i) => (
                      <Layout key={i} margin={{ bottom: 0.5, right: 0.5 }} ellipsis>
                        <Tag action={TagAction.Remove} label={option[optionName]} onClick={e => removeSelectedOption(e, option)} />
                      </Layout>
                    )) : (
                      <NoResults results={label} small />
                    )}
                </Layout>
                {dropdownOptions?.length ? (dropdownOptions
                  .filter(dropdownOption => selectedOptions && selectedOptions.every(selectedOption => selectedOption[optionId] !== dropdownOption[optionId]))
                  .map((option, i) => {
                    return (
                      <DropDownMenuItem key={i} onClick={e => selectOption(e, option)}>
                        <CoreText>{option[optionName]}</CoreText>
                      </DropDownMenuItem>
                    );
                  })
                ) : (
                  <NoResults results={label} small />
                )}
              </SimpleBarReact>
              <Layout margin={{ top: 1 }} display={Display.Flex} justifyContent={JustifyContent.End}>
                <Layout margin={{ right: 1 }}>
                  <Button type={ButtonType.Text} onClick={clearFilter}>
                    Clear
                  </Button>
                </Layout>
                <Button onClick={submitFilter}>Confirm</Button>
              </Layout>
            </DropDownMenu>
          </Transition>
        </Layout>
      </ClickOutDetector>
    </>
  );
}

export default EventsFilterDropdownLocalStorage;
