import React, { useState, useEffect } from 'react';
import { useHistory } from 'react-router-dom';
import SimpleBarReact from 'simplebar-react';
import {
  Display,
  AlignItems,
  Layout,
  JustifyContent,
  Button,
  CheckBox,
  FlexWrap,
  Modal,
  ModalHeader,
  ModalBody,
  ModalFooter,
  FormGroup,
  ButtonState,
  SVGAsset,
  ButtonType,
  Input,
  InputType
} from 'twitch-core-ui';
import Overlay from './overlay';
import EventsHeaderNav from './events-header-nav';
import EventsHeaderTabs from './events-header-tabs';
import EventsFilterDropdownChannel from './events-filter-dropdown-channel';
import EventsFilterDropdownGame from './events-filter-dropdown-game';
import EventsFilterTimeline from './events-filter-timeline';
import EventsFilterDropdown from './events-filter-dropdown';
import EventsFilterDropdownLocalStorage from './events-filter-dropdown-ls';
import { EVENT_ROUTES } from './../utils/config';
import { getTopics, getFormats } from '../utils/helpers';
import { coStreamingSettingsOptions } from '../utils/options';
import { useWindowDimensions } from '../hooks/useWindowDimensions';
import { useSelectedPermissions } from '../context/permissions-context';
import { MODAL_HEADER_AND_FOOTER_HEIGHT } from '../utils/config';
import { useDataContext } from '../context/data-context';
import { csvExportEventList } from '../utils/csvExport';
import useLocalStorage from '@rehooks/local-storage';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function EventsHeader({ activePage, loading }) {
  const history = useHistory();
  const searchParams = new URLSearchParams(history.location.search);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const { productsList, allTopicsFormatsOptions } = useDataContext();
  const { height } = useWindowDimensions();

  // product, topic, format, subformat, season options
  /* eslint-disable @typescript-eslint/no-unused-vars */
  const [productOptions, setProductOptions] = useState([]);
  const [topicOptions, setTopicOptions] = useState([]);
  const [formatOptions, setFormatOptions] = useState([]);

  // Inline Filters
  const [eventName, setEventName] = useState(searchParams.get('eventName') || '');
  const [products, setProducts] = useState(JSON.parse(localStorage.getItem('eventsFilterProducts')) || '');
  const [topics, setTopics] = useState(JSON.parse(localStorage.getItem('eventsFilterTopics')) || '');
  const [games, setGames] = useState(searchParams.get('gameName') || '');
  const [channels, setChannels] = useState(JSON.parse(localStorage.getItem('eventsFilterChannels')) || '');
  const [timeline, setTimeline] = useState(searchParams.get('timeline') || '');
  const [format, setFormat] = useState(searchParams.get('format') || '');
  const [distribution, setDistribution] = useState(searchParams.get('distribution') || '');
  const [coStreamingSettings, setCoStreamingSettings] = useState(searchParams.get('costreamingSettings') || '');
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);

  const submitFilter = e => {
    console.log(e, 'submit filter to searchParams');
  };

  useEffect(() => {
    setProductOptions(productsList);
  }, [productsList]);

  useEffect(() => {
    setTopicOptions(getTopics(allTopicsFormatsOptions));
  }, [allTopicsFormatsOptions]);

  useEffect(() => {
    if (topics && allTopicsFormatsOptions) {
      setFormatOptions(getFormats(topics, allTopicsFormatsOptions));
    }
  }, [topics, allTopicsFormatsOptions]);

  const handleFormatChange = format => {
    setFormat(format);
  };

  // const clearFilters = e => {
  //   e.preventDefault();
  //   localStorage.removeItem('eventsFilterProducts');
  //   localStorage.removeItem('eventsFilterTopics');
  //   localStorage.removeItem('eventsFilterChannels');
  //   history.push('/events');
  //   document.location.reload();
  // };

  const permissions = useSelectedPermissions();
  const canEdit = permissions.includes('Can Manage Products') || permissions.includes('Manage Owned Channels');

  const onDownloadAsCsvClicked = () => {
    csvExportEventList(+selectedContentCreator);
  }

  return (
    <>
      <Layout margin={{ bottom: 2 }}>
        <EventsHeaderTabs />
      </Layout>
      <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between} flexWrap={FlexWrap.Wrap} padding={{ bottom: 2 }}>
        <Layout display={Display.Flex}>
          <Button
            type={ButtonType.Secondary}
            state={loading ? ButtonState.Loading : ButtonState.Default}
            icon={SVGAsset.Filter}
            onClick={e => {
              e.preventDefault();
              setIsModalOpen(true);
            }}>
            Filter Listings
          </Button>
          <EventsHeaderNav activePage={activePage} dropdownOptions={EVENT_ROUTES} />
          {isModalOpen && (
            <Overlay onClickOut={() => setIsModalOpen(false)}>
              <Layout className="modal--wrapper modal--wrapper__filters">
                <Modal>
                  <form id="eventsFilter" onSubmit={submitFilter}>
                    <ModalHeader
                      title="Filter Listings"
                      closeButton={{
                        'aria-label': 'Close Modal',
                        onClick: () => setIsModalOpen(false)
                      }}></ModalHeader>
                    <SimpleBarReact style={{ maxHeight: height < 800 ? `calc(100vh - ${MODAL_HEADER_AND_FOOTER_HEIGHT})` : 660 }}>
                      <ModalBody>
                        {/* <Layout margin={{ bottom: 2 }}>
                          <Button type={ButtonType.Text} icon={SVGAsset.Trash} onClick={clearFilters}>
                            Clear All Filters
                          </Button>
                        </Layout> */}
                        <Layout margin={{ bottom: 2 }}>
                          <FormGroup label="Event Name">
                            <Input
                              type={InputType.Text}
                              name="eventName"
                              placeholder="Event Name"
                              value={eventName}
                              onChange={e => setEventName(e.currentTarget.value)}
                            />
                          </FormGroup>
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <EventsFilterDropdownLocalStorage
                            dropdownOptions={productOptions}
                            filterName="products"
                            label="Products"
                            query={products}
                            setQuery={setProducts}
                            localStorageKey="eventsFilterProducts"
                            optionName="product_name"
                            optionId="product_id"
                          />
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <EventsFilterDropdownLocalStorage
                            dropdownOptions={topicOptions}
                            filterName="topic"
                            label="Topics"
                            query={topics}
                            setQuery={setTopics}
                            localStorageKey="eventsFilterTopics"
                            optionName="topic"
                            optionId="topic_id"
                          />
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <EventsFilterDropdownGame games={games} setGames={setGames} />
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <EventsFilterDropdownChannel channels={channels} setChannels={setChannels} />
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <EventsFilterTimeline timeline={timeline} setTimeline={setTimeline} />
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <EventsFilterDropdown
                            dropdownOptions={formatOptions}
                            filterName="format"
                            label="Format"
                            query={format}
                            setQuery={query => handleFormatChange(query)}
                            disabled={!topics.length}
                            hint="Please select a topic to get format options"
                          />
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <EventsFilterDropdown
                            dropdownOptions={coStreamingSettingsOptions}
                            filterName="costreamingSettings"
                            label="Co-streaming Settings"
                            query={coStreamingSettings}
                            setQuery={setCoStreamingSettings}
                          />
                        </Layout>
                        <Layout margin={{ bottom: 2 }}>
                          <FormGroup label="Distribution Exclusivity">
                            <CheckBox
                              name="distribution"
                              label="Twitch has the exclusive right to broadcast this content."
                              checked={!!distribution}
                              onChange={e => setDistribution(e.currentTarget.checked ? 'checked' : '')}
                            />
                          </FormGroup>
                        </Layout>
                      </ModalBody>
                    </SimpleBarReact>
                    <ModalFooter
                      primaryButtonProps={{
                        children: 'Submit'
                      }}
                      secondaryButtonProps={{
                        children: 'Cancel',
                        onClick: e => {
                          e.preventDefault();
                          setIsModalOpen(false);
                        }
                      }}
                    />
                  </form>
                </Modal>
              </Layout>
            </Overlay>
          )}
        </Layout>
        <Layout display={Display.Flex}>
          <Layout margin={{right: 1}}>
            <Button disabled={!selectedContentCreator} onClick={onDownloadAsCsvClicked}>Download As CSV</Button>
          </Layout>
        {canEdit && <Button linkTo="/events/create">Create Single Event</Button>}
        </Layout>
      </Layout>
    </>
  );
}

export default EventsHeader;
