import React, { useState } from 'react';
import { StyledLayout, Background, FontSize, Table, TableHeader, TableHeading, TableBody, TableRow, TableCell, Sorting, Position } from 'twitch-core-ui';
import EventsDrawer from './events-drawer';
import { getMomentFullDateTime } from '../utils/dateTimeHelpers';
import { useSelectedEvents } from '../context/events-context';
import './styles/Table.scss';

function EventsList({ events, grouping = 'event', sortData = [], setSortData = undefined }) {
  const [activeSortHeader] = useState((sortData && sortData[0] && sortData[0].col) || '');

  const sortTableHeader = header => {
    setSortData([{ col: header.id, ord: 0, asc: header.id === activeSortHeader ? !sortData[0].asc : false }]);
  };

  return (
    <StyledLayout
      className="events--card-wrapper table--wrapper table--selection-wrapper"
      background={Background.Base}
      fontSize={FontSize.Size7}
      position={Position.Relative}>
      <Table alternateRows>
        <TableHeader>
          <TableRow>
            {[
              { name: 'Name', id: 'event_name' },
              { name: 'Product', id: 'product_name' },
              { name: 'Topic', id: 'topic' },
              { name: 'Category', id: 'game_name' },
              { name: 'Channel(s)', id: 'channel_count' },
              { name: 'Format', id: 'format' },
              { name: 'Estimated ACCV', id: 'estimated_average_ccv' },
              { name: 'Timeline', id: 'end_time' }
            ].map(header => {
              const { name, id } = header;
              const isActiveHeader = activeSortHeader !== id;
              const isActiveSortDataAscending = sortData[0] && sortData[0].asc;
              return setSortData === undefined ? (
                <TableHeading key={id} label={name} />
              ) : (
                <TableHeading
                  key={id}
                  label={name}
                  sorting={isActiveHeader ? Sorting.Default : isActiveSortDataAscending ? Sorting.Ascending : Sorting.Descending}
                  onClick={() => sortTableHeader(header)}
                />
              );
            })}
          </TableRow>
        </TableHeader>
        <TableBody>{events && events.map(targetEvent => <EventsRow key={`${grouping}-${targetEvent.event_id}`} targetEvent={targetEvent} />)}</TableBody>
      </Table>
      <EventsDrawer />
    </StyledLayout>
  );
}

export default EventsList;

function EventsRow({ targetEvent }) {
  const {
    event_id,
    event_name,
    topic,
    sub_topic,
    game_name,
    channel_count,
    estimated_average_ccv,
    start_time,
    end_time,
    product_name,
    season_name,
    format,
    sub_format
  } = targetEvent;
  const [selectedEvents, setSelectedEvents] = useSelectedEvents();
  const isSelectedEvent = selectedEvents && selectedEvents[0] && selectedEvents[0].event_id === event_id;

  const toggleSelectedEvent = targetEvent => {
    const isSelectedEventClickedAgain = selectedEvents && selectedEvents[0] && selectedEvents[0].event_id;
    if (targetEvent.event_id === isSelectedEventClickedAgain) {
      setSelectedEvents([]);
    } else {
      setSelectedEvents([targetEvent]);
    }
  };

  return (
    <TableRow className={`events--row ${isSelectedEvent ? 'selected' : ''}`} height={30} onClick={() => toggleSelectedEvent(targetEvent)}>
      <TableCell width={250}>{event_name || 'N/A'}</TableCell>
      <TableCell width={180}>
        {product_name || 'N/A'}
        {season_name ? ' - ' + season_name : ''}
      </TableCell>
      <TableCell width={180}>
        {topic || 'N/A'}
        {sub_topic ? ' - ' + sub_topic : ''}
      </TableCell>
      <TableCell width={170}>{game_name || 'N/A'}</TableCell>
      <TableCell width={115}>{channel_count || 0}</TableCell>
      <TableCell width={270}>
        {format || 'N/A'}
        {sub_format ? ' - ' + sub_format : ''}
      </TableCell>
      <TableCell width={160}>{(estimated_average_ccv && estimated_average_ccv.toLocaleString()) || 'N/A'}</TableCell>
      <TableCell width={280}>
        <span className="white-space-nowrap">{getMomentFullDateTime(start_time)}</span>&nbsp;-&nbsp;
        <span className="white-space-nowrap">{getMomentFullDateTime(end_time)}</span>
      </TableCell>
    </TableRow>
  );
}
