import React, { useState, useEffect } from 'react';
import { useHistory } from 'react-router-dom';
import { Layout, CoreText, TextType, FontWeight } from 'twitch-core-ui';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import PaginationWrapper from './pagination-wrapper';
import EventsList from './events-list';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST } from '../hooks/useFetch';
import EventsHeader from './events-header';
import EventsPlaceholder from './events-placeholder';
import MetricsConsole from './metrics-console';
import { hyphenify } from '../utils/helpers';
import { useLocalStorage } from '@rehooks/local-storage';
import { buildEventFilters } from '../utils/helpers';
import { EventsProvider } from '../context/events-context';
import './styles/Events.scss';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function EventsSeason() {
  const history = useHistory();
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const searchParams = new URLSearchParams(history.location.search);
  const [currentPage, setCurrentPage] = useState(parseInt(searchParams.get('page')) || 1);
  const [filters, setFilters] = useState(buildEventFilters(searchParams, selectedContentCreator));

  let options = {
    limit: 10,
    page: currentPage - 1,
    eventtype: 1,
    filters: filters,
    'req-cols': ['*']
  };
  const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-season-event-list`, options);
  const events = data == null || data.data == null ? null : data.data;
  const totalPages = data == null || data.data == null ? 0 : data.data.total_pages;
  const metrics = data == null || data.metrics == null ? [] : data.metrics;

  useEffect(() => {
    setFilters(buildEventFilters(searchParams, selectedContentCreator)); /* eslint-disable react-hooks/exhaustive-deps */
  }, [selectedContentCreator]);

  useEffect(() => {
    setCurrentPage(currentPage);
    let params = {};
    for (let param of searchParams as any) {
      if (param[1].length) {
        params[param[0]] = param[1];
      }
    }
    if (currentPage !== 1 && params['page'] === undefined) {
      params['page'] = currentPage;
    } else if (params['page'] !== undefined) {
      params['page'] = currentPage;
    }
    history.push({
      search: new URLSearchParams(params).toString()
    });
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, [currentPage, history, filters]);

  return (
    <>
      <Layout className="events">
        <EventsHeader activePage="Season" loading={loading} />
        {error && <ErrorAlert error={error} />}
        {loading ? (
          <EventsPlaceholder type="season" />
        ) : events && events.groupings && events.groupings.length === 0 ? (
          <NoResults results="Season Events" />
        ) : (
          events &&
          events.groupings &&
          events.groupings.map(data => {
            const { season, events } = data;
            return (
              <Layout key={season}>
                <SeasonHeader season={season} />
                <Layout margin={{ bottom: 3 }}>
                  {events && events.length === 0 ? (
                    <NoResults results={`${season} events`} />
                  ) : (
                    <EventsProvider>
                      <EventsList events={events} grouping={hyphenify(season)} />
                    </EventsProvider>
                  )}
                </Layout>
              </Layout>
            );
          })
        )}
        {totalPages > 1 && <PaginationWrapper currentPage={currentPage} totalPages={totalPages} setCurrentPage={setCurrentPage} />}
      </Layout>
      <MetricsConsole metrics={metrics} loading={loading} name="Events Season" />
    </>
  );
}

function SeasonHeader({ season }) {
  return (
    <Layout margin={{ bottom: 2 }}>
      <CoreText type={TextType.H3} fontWeight={FontWeight.SemiBold}>
        {season}
      </CoreText>
    </Layout>
  );
}

export default EventsSeason;
