import React, { useState, useRef } from 'react';
import { useHistory } from 'react-router-dom';
import SimpleBarReact from 'simplebar-react';
import Cookies from 'js-cookie';
import { v4 as uuidv4 } from 'uuid';
import {
  Layout,
  CoreText,
  TextType,
  Display,
  AlignItems,
  JustifyContent,
  Button,
  ButtonType,
  SVGAsset,
  ComboInput,
  InputType,
  CoreButtonType,
  Position,
  SplitButton,
  CoreButtonDropdownType,
  DropDownMenu,
  BalloonSize,
  DropDownMenuItem,
  Modal,
  ModalHeader,
  ModalBody,
  ModalFooter,
  FontSize,
  ModalSize,
  FormGroup,
  SearchInput,
  StyledLayout,
  Table,
  TableBody,
  TableRow,
  TableCell,
  TableHeader,
  TableHeading,
  Tooltip
} from 'twitch-core-ui';
import { useSelectedEvents } from '../context/events-context';
import { useEventsWorkspaceList } from '../context/events-workspace-context';
import { useWindowDimensions } from '../hooks/useWindowDimensions';
import { useFetchPOST } from '../hooks/useFetch';
import { MODAL_HEADER_AND_FOOTER_HEIGHT } from '../utils/config';
import { ATLAS_BASE_URL } from '../utils/config';
import { getMomentFullDateTime } from '../utils/dateTimeHelpers';
import { useLocalStorage } from '@rehooks/local-storage';
import { fetchWithTimeout } from '../utils/helpers';
import { getDefaultTimezoneOption } from '../utils/dateTimeHelpers';
import useUpdateEffect from '../hooks/useUpdateEffect';
import EventsHeaderTabs from './events-header-tabs';
import ClickOutDetector from './click-out-detector';
import Overlay from './overlay';
import Loading from './loading';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import './styles/Table.scss';
import { CSVFileUploadModal } from './csv-file-upload-modal';
import { Portal } from '../utils/portal';
import { AtlasEvent } from '../utils/types';
import { EditEventChannelsModal } from './edit-event-channels-modal';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function EventsWorkspaceHeader() {
  const history = useHistory();
  const { eventsWorkspaceList, setEventsWorkspaceList, addEventsToWorkspaceList } = useEventsWorkspaceList();
  const [selectedEvents, setSelectedEvents] = useSelectedEvents();
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const [copies, setCopies] = useState('1');
  const [isExistingEventDropdownDisplayed, setIsExistingEventDropdownDisplayed] = useState(false);
  const [isFileUploadModalDisplayed, setIsFileUploadModalDisplayed] = useState(false);
  const [isExistingEventModalDisplayed, setIsExistingEventModalDisplayed] = useState(false);
  const [isEditChannelsModalDisplayed, setIsEditChannelsModalDisplayed] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const { height } = useWindowDimensions();

  const [errorMessage, setErrorMessage] = useState('');
  const [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-event-list`, {
    limit: 10,
    page: 0,
    eventtype: 1,
    'req-cols': ['*'],
    filters: [
      { 'filter-type': 4, key: 'event_name', v1: searchQuery },
      { 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }
    ],
    sortdata: [{ col: 'event_name', asc: true }]
  });
  const timeoutRef = useRef(null);

  useUpdateEffect(() => {
    if (timeoutRef.current !== null) {
      clearTimeout(timeoutRef.current);
    }
    timeoutRef.current = setTimeout(() => {
      timeoutRef.current = null;
      fetchData();
    }, 300);
    return () => {
      timeoutRef.current = false;
    };
  }, [searchQuery]);

  const toggleSelectedEvent = targetEvent => {
    const isSelectedEventClickedAgain = selectedEvents && selectedEvents[0] && selectedEvents[0].event_id;
    if (targetEvent.event_id === isSelectedEventClickedAgain) {
      setSelectedEvents([]);
    } else {
      setSelectedEvents([targetEvent]);
    }
  };

  const removeCopyText = text => {
    if (text.includes(' COPY')) {
      const copyTextIndex = text.indexOf(' COPY');
      return text.slice(0, copyTextIndex);
    } else {
      return text;
    }
  };

  const copySelectionToWorkspace = () => {
    const eventCopyObj = (({ event_id, ...o }) => o)(selectedEvents[0]);
    eventCopyObj['event_name'] = removeCopyText(eventCopyObj['event_name']) + ` COPY(${eventsWorkspaceList.length + 1})`;
    eventCopyObj['workspace_id'] = uuidv4();

    const eventsWorkspaceArr = [...eventsWorkspaceList, eventCopyObj];
    localStorage.setItem('eventsWorkspaceList', JSON.stringify(eventsWorkspaceArr));
    setIsExistingEventModalDisplayed(false);
    setEventsWorkspaceList(eventsWorkspaceArr);
    setSelectedEvents([]);
  };

  const copySelectedEvent = () => {
    const newCopiesArr = [];

    for (let i = 0; i < parseInt(copies); i++) {
      const copy = Object.assign({}, selectedEvents[0]);
      copy['event_name'] = removeCopyText(copy['event_name']) + ` COPY(${eventsWorkspaceList.length + 1 + i})`;
      copy['workspace_id'] = uuidv4();
      newCopiesArr.push(copy);
    }

    const eventsWorkspaceArr = [...eventsWorkspaceList, ...newCopiesArr];
    setEventsWorkspaceList(eventsWorkspaceArr);
    localStorage.setItem('eventsWorkspaceList', JSON.stringify(eventsWorkspaceArr));
  };

  const deleteSelectedEvents = () => {
    let filteredEvents = [];
    const checkboxList: any = document.querySelectorAll('.tw-checkbox__input:not([name="events-workspace-select-all"])');
    for (let i = 0; i < checkboxList.length; i++) {
      if (checkboxList[i].checked) {
        checkboxList[i].checked = false;
      } else {
        filteredEvents.push(eventsWorkspaceList[i]);
      }
    }
    setSelectedEvents([]);
    localStorage.setItem('eventsWorkspaceList', JSON.stringify(filteredEvents));
    setEventsWorkspaceList(filteredEvents);
  };

  const openFileUploadPrompt = () => {
    setIsFileUploadModalDisplayed(true);
  }

  const onChannelsUpdated = () => {
    // Save updated items to local storage after updating channels
    localStorage.setItem('eventsWorkspaceList', JSON.stringify(eventsWorkspaceList));
  }

  const publishSelectedEvents = () => {
    const existingEventNames = [];
    for (let item of data.data.items) {
      existingEventNames.push(item.event_name);
    }

    let optionsArr = [];
    for (let i = 0; i < selectedEvents.length; i++) {
      if (existingEventNames.indexOf(selectedEvents[i].event_name) > -1) {
        setErrorMessage(`"${selectedEvents[i].event_name}" already exists in Events. Please revise selected event name before publishing.`);
        return;
      }

      let obj = {
        event: selectedEvents[i],
        streams: selectedEvents[i].channels,
        time_zone: localStorage.getItem('timezone') === null ? getDefaultTimezoneOption().value : JSON.parse(localStorage.getItem('timezone')).value
      };

      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${Cookies.get('token')}`
        },
        body: JSON.stringify(obj)
      };

      optionsArr.push(fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-event-with-streams`, options));
    }

    Promise.all(optionsArr)
      .then(function(responses) {
        return Promise.all(
          responses.map(function(response) {
            return response.json();
          })
        );
      })
      .then(function(data) {
        console.log(`/edit-event:`, data);
      })
      .catch(function(err) {
        console.error(`/edit-event:`, err);
        setErrorMessage(err);
      });

    // TODO: create separate function that publishes and deletes when published individually
    // also highlight duplicate event name to clearly indicate error
    deleteSelectedEvents();
    history.push('/events');
  };

  const onCSVUploaded = (results: AtlasEvent[]) => {
    if (results) {
      addEventsToWorkspaceList(results);
    }
  }

  return (
    <Layout className="events-workspace--header">
      {errorMessage && <ErrorAlert error={errorMessage} setError={setErrorMessage} />}
      <Layout margin={{ bottom: 2 }}>
        <EventsHeaderTabs />
      </Layout>
      <Layout margin={{ bottom: 0.5 }}>
        <CoreText type={TextType.H3}>Events Creation Workspace</CoreText>
      </Layout>
      <Layout margin={{ bottom: 3 }}>
        <CoreText>
          Welcome to the Event Creation Workspace! You may use this space to create, copy, edit, and delete new events. The <strong>Add New Event</strong>{' '}
          button will allow you to create a new event from scratch, or by using an existing event from the listing page. <strong>Copy Selected Event</strong>{' '}
          allows you to duplicate the selected event any number of times, and you may delete unwanted events by selecting them and clicking the{' '}
          <strong>Delete Selected Event(s)</strong> button. All fields are editable in this workspace.
        </CoreText>
      </Layout>
      <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between} margin={{ bottom: 1 }}>
        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
          <Layout margin={{ right: 1 }} position={Position.Relative}>
            <SplitButton
              type={CoreButtonType.Secondary}
              icon={SVGAsset.Recommended}
              dropdown={{ type: CoreButtonDropdownType.ArrowDown, onClick: () => setIsExistingEventDropdownDisplayed(true) }}
              linkTo="/events/create">
              Add New Event
            </SplitButton>
            <ClickOutDetector onClickOut={() => setIsExistingEventDropdownDisplayed(false)}>
              <DropDownMenu show={isExistingEventDropdownDisplayed} size={BalloonSize.Auto}>
                <DropDownMenuItem onClick={() => setIsExistingEventModalDisplayed(true)}>
                  <CoreText fontSize={FontSize.Size7}>Add From Existing Event</CoreText>
                </DropDownMenuItem>
                <DropDownMenuItem onClick={() => openFileUploadPrompt()}>
                  <CoreText fontSize={FontSize.Size7}>Add From CSV File</CoreText>
                </DropDownMenuItem>
              </DropDownMenu>
            </ClickOutDetector>
            {isFileUploadModalDisplayed && 
              <Portal>
                <CSVFileUploadModal 
                  onCancel={() => setIsFileUploadModalDisplayed(false)}
                  onDone={(results) => {
                    setIsFileUploadModalDisplayed(false);
                    onCSVUploaded(results);
                  }}
                />
              </Portal>
            }
            {isEditChannelsModalDisplayed &&
              <Portal>
                <EditEventChannelsModal 
                  onCancel={() => setIsEditChannelsModalDisplayed(false)}
                  onDone={() => {
                    setIsEditChannelsModalDisplayed(false);
                    onChannelsUpdated();
                  }}
                  events={selectedEvents}
                />
              </Portal>
            }
            {isExistingEventModalDisplayed && (
              <Overlay onClickOut={() => setIsExistingEventModalDisplayed(false)}>
                <Layout className="modal--wrapper condensed">
                  <Modal size={ModalSize.Medium}>
                    <ModalHeader
                      title="Select an Existing Event"
                      closeButton={{
                        'aria-label': 'Close Modal',
                        onClick: () => setIsExistingEventModalDisplayed(false)
                      }}
                    />
                    <SimpleBarReact style={{ maxHeight: height < 800 ? `calc(100vh - ${MODAL_HEADER_AND_FOOTER_HEIGHT})` : 660 }}>
                      <ModalBody>
                        <Layout margin={{ bottom: 2 }}>
                          <FormGroup label="Search by Event Name">
                            <SearchInput placeholder="Search by Name" onChange={e => setSearchQuery(e.currentTarget.value)} />
                          </FormGroup>
                        </Layout>
                        {error && <ErrorAlert error={error} />}
                        {loading ? (
                          <Loading />
                        ) : data && data.data && data.data.items.length === 0 ? (
                          <NoResults results="events" />
                        ) : (
                          <StyledLayout className="table--selection-wrapper" fontSize={FontSize.Size7} position={Position.Relative}>
                            <Table alternateRows>
                              <TableHeader>
                                <TableRow>
                                  {['Name', 'Timeline'].map(name => (
                                    <TableHeading key={name}>{name}</TableHeading>
                                  ))}
                                </TableRow>
                              </TableHeader>
                              <TableBody>
                                {data &&
                                  data.data &&
                                  data.data.items.map(targetEvent => {
                                    const { event_id, event_name, start_time, end_time } = targetEvent;
                                    const isSelectedEvent = selectedEvents && selectedEvents[0] && selectedEvents[0].event_id === event_id;
                                    return (
                                      <TableRow
                                        className={`events--row ${isSelectedEvent ? 'selected' : ''}`}
                                        key={event_id}
                                        height={30}
                                        onClick={() => toggleSelectedEvent(targetEvent)}>
                                        <TableCell className="events--row__name">{event_name || 'N/A'}</TableCell>
                                        <TableCell>{`${getMomentFullDateTime(start_time)} - ${getMomentFullDateTime(end_time)}` || 'N/A'}</TableCell>
                                      </TableRow>
                                    );
                                  })}
                              </TableBody>
                            </Table>
                          </StyledLayout>
                        )}
                      </ModalBody>
                    </SimpleBarReact>
                    <ModalFooter
                      primaryButtonProps={{
                        children: 'Copy Selection to Workspace',
                        onClick: copySelectionToWorkspace
                      }}
                      secondaryButtonProps={{
                        children: 'Cancel',
                        onClick: e => {
                          e.preventDefault();
                          setIsExistingEventModalDisplayed(false);
                        }
                      }}
                    />
                  </Modal>
                </Layout>
              </Overlay>
            )}
          </Layout>
          <Layout margin={{ right: 1 }}>
            <Tooltip label="This action is only available when 1 event is selected" disabled={selectedEvents && selectedEvents.length === 1}>
              <ComboInput
                type={InputType.Number}
                value={copies}
                disabled={selectedEvents && selectedEvents.length !== 1}
                onChange={e => setCopies(e.currentTarget.value)}
                min={0}
                max={100}
                buttonProps={{
                  children: 'Copy Selected Event',
                  type: CoreButtonType.Secondary,
                  'aria-label': 'Copy',
                  onClick: copySelectedEvent,
                  icon: SVGAsset.Copy,
                  disabled: (selectedEvents && selectedEvents.length !== 1) || parseInt(copies) < 1 || !copies.length
                }}
              />
            </Tooltip>
          </Layout>
          <Layout margin={{ right: 1 }}>
            <Button type={ButtonType.Secondary} icon={SVGAsset.Trash} disabled={selectedEvents && selectedEvents.length === 0} onClick={deleteSelectedEvents}>
              Delete Selected Event{selectedEvents.length > 1 ? 's' : ''}
            </Button>
          </Layout>
          <Layout margin={{ right: 1 }}>
            <Button type={ButtonType.Secondary} icon={SVGAsset.Account} disabled={selectedEvents && selectedEvents.length === 0} onClick={() => setIsEditChannelsModalDisplayed(true)}>
              Edit Channels
            </Button>
          </Layout>
        </Layout>
        <Button icon={SVGAsset.Upload} disabled={selectedEvents && selectedEvents.length === 0} onClick={publishSelectedEvents}>
          Publish Selected Event{selectedEvents.length > 1 ? 's' : ''}
        </Button>
      </Layout>
    </Layout>
  );
}

export default EventsWorkspaceHeader;
