import React from 'react';
import {
  Layout,
  StyledLayout,
  Background,
  FontSize,
  Table,
  TableHeader,
  TableHeading,
  TableBody,
  TableRow,
  CheckBox,
  TextAlign,
  CoreText,
  CoreLink,
  BorderRadius
} from 'twitch-core-ui';
import EventsWorkspaceRow from './events-workspace-row';
import LabelRequired from './label-required';
import { useSelectedEvents } from '../context/events-context';
import { useEventsWorkspaceList } from '../context/events-workspace-context';
import './styles/Events.scss';
import { useDataContext } from '../context/data-context';
import { WorkspaceEvent } from '../utils/types';

function EventsWorkspaceList() {
  const [selectedEvents, setSelectedEvents] = useSelectedEvents();
  const { eventsWorkspaceList } = useEventsWorkspaceList();
  // format and subformat options
  const { allTopicsFormatsOptions, productsList } = useDataContext();

  const selectAllChange = e => {
    let itemName = e.target.name;
    let checked = e.target.checked;
    const checkboxList: any = document.querySelectorAll('.tw-checkbox__input:not([name="events-workspace-select-all"])');
    if (itemName === 'events-workspace-select-all') {
      for (let checkbox of checkboxList) {
        if (checked) {
          checkbox.checked = true;
          checkbox.closest('.events--row').classList.add('selected');
          setSelectedEvents(JSON.parse(localStorage.getItem('eventsWorkspaceList')));
        } else {
          checkbox.checked = false;
          checkbox.closest('.events--row').classList.remove('selected');
          setSelectedEvents([]);
        }
      }
    }
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>, row: WorkspaceEvent) => {
    const { target } = e;
    const { name, checked } = target;
    if (checked) {
      target.closest('.events--row').classList.add('selected');
      setSelectedEvents(selectedEvents => selectedEvents.concat(row));
    } else {
      // uncheck selectAll input
      const checkboxSelectAll: any = document.querySelector('.tw-checkbox__input[name="events-workspace-select-all"]');
      checkboxSelectAll.checked = false;

      // uncheck selected input row
      target.closest('.events--row').classList.remove('selected');
      const filteredEvents = selectedEvents.filter(selectedEvent => selectedEvent.workspace_id !== name);
      setSelectedEvents(filteredEvents);
    }
  };

  return (
    <Layout padding={{ top: 2 }}>
      <StyledLayout className="events-workspace-list-wrapper table--selection-wrapper" background={Background.Base} fontSize={FontSize.Size7}>
        <Table alternateRows>
          <TableHeader>
            <TableRow>
              <TableHeading textAlign={TextAlign.Center}>
                <CheckBox
                  name="events-workspace-select-all"
                  label="&nbsp;"
                  onChange={selectAllChange}
                  checked={eventsWorkspaceList.length > 0 && selectedEvents.length === eventsWorkspaceList.length}
                  disabled={eventsWorkspaceList && eventsWorkspaceList.length === 0}
                />
              </TableHeading>
              {[
                { name: 'Name', required: true },
                { name: 'Start Date', required: true },
                { name: 'Start Time', required: true },
                { name: 'End Date', required: true },
                { name: 'End Time', required: true },
                { name: 'Estimated ACCV', required: false },
                { name: 'Product', required: true },
                { name: 'Format', required: true },
              ].map(tableHeader => {
                const { name, required } = tableHeader;
                return (
                  <TableHeading key={name}>
                    <CoreText ellipsis>
                      {required ? <LabelRequired /> : null}
                      {name}
                    </CoreText>
                  </TableHeading>
                );
              })}
            </TableRow>
          </TableHeader>
          <TableBody>
            {eventsWorkspaceList.length
              ? eventsWorkspaceList.map(workspaceEvent => (
                  <EventsWorkspaceRow
                    key={workspaceEvent && workspaceEvent.workspace_id}
                    workspaceEvent={workspaceEvent}
                    handleChange={handleChange}
                    allTopicsFormatsOptions={allTopicsFormatsOptions}
                    productsList={productsList}
                  />
                ))
              : null}
          </TableBody>
        </Table>
        {eventsWorkspaceList.length === 0 ? (
          <StyledLayout padding={{ x: 1, y: 2 }} fullWidth border borderRadius={BorderRadius.Medium}>
            <CoreText>
              Your workspace is currently empty. Click the{' '}
              <CoreLink linkTo="/events/create">
                <strong>Add New Event</strong>
              </CoreLink>{' '}
              button to get started!
            </CoreText>
          </StyledLayout>
        ) : null}
      </StyledLayout>
    </Layout>
  );
}

export default EventsWorkspaceList;
