import React, { useState, useEffect, useRef } from 'react';
import { TableRow, TableCell, CheckBox, Input, InputType, Select, Layout, SVG, SVGAsset, Display, Tooltip, TooltipDirection } from 'twitch-core-ui';
import DatePicker from './date-picker';
import FormInputTime from './form-input-time';
import { getMomentDate, getMomentTime, createDateString, setMomentDateTime } from '../utils/dateTimeHelpers';
import { useEventsWorkspaceList } from '../context/events-workspace-context';
import useUpdateEffect from '../hooks/useUpdateEffect';
import { Format, Product, TopicOptions, WorkspaceEvent } from '../utils/types';

interface Props {
  workspaceEvent: WorkspaceEvent;
  handleChange: (e: React.ChangeEvent<HTMLInputElement>, row: WorkspaceEvent) => void;
  productsList: Product[];
  allTopicsFormatsOptions: TopicOptions[];
}

function EventsWorkspaceRow(props: Props) {
  const { workspaceEvent, handleChange, allTopicsFormatsOptions, productsList } = props;
  const { eventsWorkspaceList } = useEventsWorkspaceList();

  const [workspaceId, setEventsWorkspaceId] = useState(workspaceEvent.workspace_id);
  const [eventRow, setEventRow] = useState(workspaceEvent);
  const [eventName, setEventName] = useState(workspaceEvent.event_name || '');
  const [startDate, setStartDate] = useState(workspaceEvent.start_time ? new Date(getMomentDate(workspaceEvent.start_time)) : null);
  const [startTime, setStartTime] = useState(workspaceEvent.start_time ? getMomentTime(workspaceEvent.start_time) : '');
  const [isInvalidStartTime, setIsInvalidStartTime] = useState(false);
  const [endDate, setEndDate] = useState(workspaceEvent.end_time ? new Date(getMomentDate(workspaceEvent.end_time)) : null);
  const [endTime, setEndTime] = useState(workspaceEvent.end_time ? getMomentTime(workspaceEvent.end_time) : '');
  const [isInvalidEndTime, setIsInvalidEndTime] = useState(false);
  const [estimatedAverageCCV, setEstimatedAverageCCV] = useState(workspaceEvent.estimated_average_ccv);
  const [format, setFormat] = useState(workspaceEvent.format || '');
  const [productName, setProductName] = useState(workspaceEvent.product_name || '');
  const [formatOptions, setFormatOptions] = useState<Format[]>([]);

  const timeoutRef = useRef(null);

  useEffect(() => {
    setEventRow(workspaceEvent);
  }, [workspaceEvent]);

  function changeFormatOptions(topic: string) {
    if (topic && allTopicsFormatsOptions) {
      const options = allTopicsFormatsOptions.find(a => a.topic === topic);
      if (options) {
        setFormatOptions(options.formats);
      }
    }
  }

  useEffect(() => {
    if (eventRow != null) {
      const { workspace_id, event_name, start_time, end_time, estimated_average_ccv, format, product_name } = eventRow;
      setEventsWorkspaceId(workspace_id || '');
      setEventName(event_name || '');
      setStartDate(start_time ? new Date(getMomentDate(start_time)) : null);
      setStartTime(start_time ? getMomentTime(start_time) : '');
      setEndDate(end_time ? new Date(getMomentDate(end_time)) : null);
      setEndTime(end_time ? getMomentTime(end_time) : '');
      setEstimatedAverageCCV(estimated_average_ccv);
      setFormat(format || '');
      setProductName(product_name || '');
    } /* eslint-disable react-hooks/exhaustive-deps */
  }, [eventRow]);

  useUpdateEffect(() => {
    setTimeout(() => {
      eventRow.event_name = eventName;
      updateEventRowAndList(eventRow);
    }, 300);
  }, [eventName]);

  useUpdateEffect(() => {
    if (startDate == null || startTime == null) return;
    if (startTime.length >= 5) {
      const dateStr = createDateString(startDate.getFullYear(), startDate.getMonth() + 1, startDate.getDate());
      eventRow.start_time = setMomentDateTime(dateStr + 'T' + startTime);
      updateEventRowAndList(eventRow);
    }
  }, [startDate, startTime]);

  useUpdateEffect(() => {
    if (endDate == null || endTime == null) return;
    if (endTime.length >= 5) {
      const dateStr = createDateString(endDate.getFullYear(), endDate.getMonth() + 1, endDate.getDate());
      eventRow.end_time = setMomentDateTime(dateStr + 'T' + endTime);
      updateEventRowAndList(eventRow);
    }
  }, [endDate, endTime]);

  useUpdateEffect(() => {
    setTimeout(() => {
      eventRow.estimated_average_ccv = estimatedAverageCCV;
      updateEventRowAndList(eventRow);
    }, 300);
  }, [estimatedAverageCCV]);

  useUpdateEffect(() => {
    eventRow.format = format;
    updateEventRowAndList(eventRow);
  }, [format]);

  useEffect(() => {
    if (!productsList) {
      return;
    }

    const product = productsList.find(a => a.product_name === productName);
    if (!product) {
      console.error("Could not set product: No product found for matching name");
      return;
    }

    eventRow.product_name = productName;
    eventRow.product_id = product.product_id;
    eventRow.topic = product.topic;
    changeFormatOptions(eventRow.topic);
    updateEventRowAndList(eventRow);
  }, [productName, productsList, allTopicsFormatsOptions]);

  const updateEventRowAndList = eventRow => {
    if (timeoutRef.current !== null) {
      clearTimeout(timeoutRef.current);
    }
    timeoutRef.current = setTimeout(() => {
      timeoutRef.current = null;
      setEventRow(eventRow);
      localStorage.setItem('eventsWorkspaceList', JSON.stringify(eventsWorkspaceList));
    }, 300);
  };

  const handleEventsWorkspaceRowChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    handleChange(e, eventRow);
    // setIsEventsWorkspaceRowActive(!isEventsWorkspaceRowActive);
  };

  return (
    <TableRow className="events--row">
      <TableCell width={50}>
        <Layout display={Display.Flex}>
          <CheckBox name={workspaceId} label="&nbsp;" onChange={handleEventsWorkspaceRowChange} />
          {startDate > endDate ? (
            <Tooltip label="Are you sure you want the Start Date after the End Date?" direction={TooltipDirection.Right}>
              <SVG asset={SVGAsset.Warning} />
            </Tooltip>
          ) : null}
        </Layout>
      </TableCell>
      <TableCell width={280}>
        <Input type={InputType.Text} value={eventName} onChange={e => setEventName(e.currentTarget.value)} required error={!eventName} />
      </TableCell>
      <TableCell width={150}>
        <DatePicker defaultDate={startDate} onChange={setStartDate} required />
      </TableCell>
      <TableCell width={150}>
        <FormInputTime time={startTime} setTime={setStartTime} isInvalidTime={isInvalidStartTime} setIsInvalidTime={setIsInvalidStartTime} required />
      </TableCell>
      <TableCell width={150}>
        <DatePicker defaultDate={endDate} onChange={setEndDate} required />
      </TableCell>
      <TableCell width={150}>
        <FormInputTime time={endTime} setTime={setEndTime} isInvalidTime={isInvalidEndTime} setIsInvalidTime={setIsInvalidEndTime} required />
      </TableCell>
      <TableCell width={130}>
        <Input
          type={InputType.Number}
          placeholder="Estimated Average CCV"
          value={estimatedAverageCCV?.toString()}
          onChange={e => setEstimatedAverageCCV(+e.currentTarget.value)}
        />
      </TableCell>

      <TableCell width={200}>
        <Select value={productName} onChange={e => setProductName(e.currentTarget.value)} required error={!productName}>
          <option disabled value="">
            Select Product
          </option>
          {productsList &&
            productsList.map(option => (
              <option value={option.product_name} key={`product-${option.product_name}`}>
                {option.product_name}
              </option>
            )
          )}
        </Select>
      </TableCell>

      <TableCell width={200}>
        <Select value={format} onChange={e => setFormat(e.currentTarget.value)} required error={!format}>
          <option disabled value="">
            Select Format
          </option>
          {formatOptions &&
            formatOptions.map(option => (
              <option value={option.format} key={`format-${option.format}`}>
                {option.format}
              </option>
            ))}
        </Select>
      </TableCell>

    </TableRow>
  );
}

export default EventsWorkspaceRow;
