import * as React from 'react';
import {
  DropZone,
  StyledLayout,
  Background,
  InjectLayout,
  Position,
  ZIndex,
  SVG,
  SVGAsset,
  SVGType,
  CoreImage,
  Layout,
  Display,
  Button,
  ButtonType,
  Tooltip
} from 'twitch-core-ui';
import './styles/FilePicker.scss';

export const FILE_INPUT_SELECTOR = 'file-picker-input';

export interface PublicProps {
  allowedFileTypes: string[];
  multiFile?: boolean;
  file: string;
  setFile: React.Dispatch<any>;
  errorMessage: string;
  setErrorMessage: React.Dispatch<React.SetStateAction<string>>;
}

export interface State {
  isDraggingOver: boolean;
}

/**
 * A file picker that supports a traditional modal for picking files as well as drag and drop functionality.
 * It is up to the consumer of this component to provide a UI representation of this functionality via React's
 * "children" component property.
 */
class FilePicker extends React.Component<PublicProps, State> {
  public state: State = { isDraggingOver: false };

  public render() {
    const { file } = this.props;
    return (
      <Layout display={Display.Flex}>
        <StyledLayout className="drag-and-drop-file-picker__wrapper" position={Position.Relative} background={Background.Alt}>
          <DropZone dragOver={this.state.isDraggingOver} error={!!this.props.errorMessage}>
            <InjectLayout position={Position.Absolute} attachTop attachLeft fullWidth fullHeight zIndex={ZIndex.Above}>
              <input
                data-a-target="file-picker-input"
                data-test-selector={FILE_INPUT_SELECTOR}
                className="drag-and-drop-file-picker__input"
                accept={this.props.allowedFileTypes.join(',')}
                multiple={this.props.multiFile || false}
                onChange={this.onFileInputChange}
                type="file"
                onDragLeave={this.onDragLeave}
                onDragEnter={this.onDragEnter}
                onDrop={this.onFileDrop}
              />
            </InjectLayout>
            {file ? (
              <CoreImage src={file} alt="File Picker Image" sizes={[{ size: '100vw' }, { size: '100vh' }]} />
            ) : (
              <SVG asset={SVGAsset.Plus} type={SVGType.Placeholder} height={40} width={40} />
            )}
          </DropZone>
        </StyledLayout>
        {file && (
          <Layout margin={{ left: 1 }}>
            <Tooltip label="Remove File">
              <Button
                type={ButtonType.Text}
                onClick={() => {
                  this.props.setFile(null);
                  this.props.setErrorMessage('');
                }}>
                <SVG asset={SVGAsset.Trash} type={SVGType.Brand} />
              </Button>
            </Tooltip>
          </Layout>
        )}
      </Layout>
    );
  }

  private onFilesSubmitted = (files: File[]) => {
    const file = files[0];
    const reader = new FileReader();

    // Max File Size: 100KB;
    if (file.size > 100000) {
      this.props.setErrorMessage('File Size can not exceed 100KB');
      // this.props.setFile(null);
    } else {
      this.props.setErrorMessage('');
    }

    // File Format: JPG or PNG
    if (!(file.type === 'image/jpeg' || file.type === 'image/png')) {
      this.props.setErrorMessage('Image type must be .jpg or .png file');
      // this.props.setFile(null);
    } else {
      this.props.setErrorMessage('');
    }

    reader.onloadend = () => {
      let img = new Image();

      img.onload = () => {
        // Dimensions: 280x156;
        if (img.width !== 280 && img.height !== 156) {
          this.props.setErrorMessage('Image Dimensions must be 280 x 156');
          // this.props.setFile(null);
        } else {
          this.props.setErrorMessage('');
        }
      };

      // if (this.props.errorMessage.length === 0) {
      img.src = reader.result as string;
      this.props.setFile(reader.result as string);
      // }
    };

    reader.readAsDataURL(file);
  };

  private onDragEnter = (e: React.DragEvent<HTMLElement>) => {
    e.preventDefault();
    this.setState({ isDraggingOver: true });
  };

  private onDragLeave = (e: React.DragEvent<HTMLElement>) => {
    e.preventDefault();
    this.setState({ isDraggingOver: false });
  };

  private onFileInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    // var fr = new FileReader();

    // fr.onload = function() {};
    e.preventDefault();
    this.finalizeSelections(e.target.files);
    e.target.value = '';
  };

  private onFileDrop = (e: React.DragEvent<HTMLElement>) => {
    e.preventDefault();
    this.setState({ isDraggingOver: false });
    this.finalizeSelections(e.dataTransfer.files);
  };

  private finalizeSelections(files: FileList | null) {
    if (files) {
      this.onFilesSubmitted(Array.from(files));
    }
  }
}

export default FilePicker;
