import './styles/Form.scss';

import React, {
  useEffect,
  useState,
} from 'react';

import SimpleBarReact from 'simplebar-react';
import {
  AlignContent,
  AlignItems,
  Avatar,
  Background,
  BorderRadius,
  ButtonIcon,
  Color,
  CoreText,
  Display,
  FlexWrap,
  FontSize,
  FontWeight,
  FormGroup,
  FormGroupOrientation,
  JustifyContent,
  Layout,
  Position,
  Select,
  StyledLayout,
  SVGAsset,
} from 'twitch-core-ui';

import { getMomentDate } from '../utils/dateTimeHelpers';
import {
  channelOwnershipOptions,
  channelTypeOptions,
  channelVerticalOptions,
} from '../utils/options';
import { Channel } from '../utils/types';
import ChannelDropdown from './channel-dropdown';
import DatePicker from './date-picker';
import LabelRequired from './label-required';
import NoResults from './no-results';

function FormChannelsSelect({ channels, setChannels, channelType = '' }) {
  const [isChannelSearchFocused, setIsChannelSearchFocused] = useState(false);

  useEffect(() => {
    // cleanup channels
    return () => setChannels([]); /* eslint-disable react-hooks/exhaustive-deps */
  }, []);

  const handleAddChannel = (e, channel: Channel) => {
    e.preventDefault();
    
    // set the default channel_type to PCC Owned - Primary if none are set by default
    channel.channel_type = channel.channel_type ?? channelTypeOptions[0];
    setIsChannelSearchFocused(false);
    if (channelType === 'date') {
      channel.custom_start_date = null;
      channel.custom_end_date = null;
    }
    setChannels(localChannels => [...localChannels, channel]);
  };

  const handleRemoveChannel = (e, id) => {
    e.preventDefault();
    setChannels(localChannels => localChannels.filter(item => item.channel_id !== id));
  };

  const updateChannelProps = (channelId, props) => {
    const channelIndex = channels.findIndex(item => channelId === item.channel_id);
    if (channelIndex === -1) {
      console.warn(`Could not update channel with id ${channelId}`);
      return;
    }

    // create a new array so that setChannels updates its reference
    const updatedChannels = [...channels];
    const updatedChannel = { ...channels[channelIndex], ...props };
    updatedChannels[channelIndex] = updatedChannel;

    setChannels(updatedChannels);
  };

  const handleChannelTypeSelect = (type, channelId) => {
    console.log(`${channelId}: ${type}`);
    updateChannelProps(channelId, { channel_type: type });
  }

  const handleChannelOwnershipSelect = (ownership, channelId) => {
    console.log(`${channelId}: ${ownership}`);
    updateChannelProps(channelId, { channel_ownership: ownership });
  }

  const handleChannelVerticalSelect = (vertical, channelId) => {
    console.log(`${channelId}: ${vertical}`);
    updateChannelProps(channelId, { channel_vertical: vertical });
  }

  const handleDate = (startDate, channelId, endDate) => {
    console.log(`${channelId}: ${startDate}-${endDate}`);

    // make final new array of objects by combining updated object.
    setChannels(localChannels => {
      const channelIndex = localChannels.findIndex(channel => channelId === channel.channel_id);

      // make new object of updated object.
      let updatedChannel = {
        ...localChannels[channelIndex],
        custom_start_date: startDate != null ? new Date(startDate).toISOString() : localChannels[channelIndex].custom_start_date,
        custom_end_date: endDate != null ? new Date(endDate).toISOString() : localChannels[channelIndex].custom_end_date
      };
      return [...localChannels.slice(0, channelIndex), updatedChannel, ...localChannels.slice(channelIndex + 1)];
    });
  };

  return (
    <>
      <Layout margin={{ bottom: 2 }}>
        <FormGroup
          orientation={!!channelType ? FormGroupOrientation.Horizontal : FormGroupOrientation.Vertical}
          label="Channels"
          hint="Search and select all channels included in this event"
          required>
          <ChannelDropdown
            channels={channels}
            isChannelSearchFocused={isChannelSearchFocused}
            setIsChannelSearchFocused={setIsChannelSearchFocused}
            handleAddChannel={handleAddChannel}
          />
        </FormGroup>
      </Layout>
      {channelType === 'date' ? (
        <ChannelsDate channels={channels} handleRemoveChannel={handleRemoveChannel} handleDate={handleDate} />
      ) : channelType === 'type' ? (
        <ChannelsType channels={channels} handleRemoveChannel={handleRemoveChannel} handleTypeSelect={handleChannelTypeSelect} />
      ) : (
        <ChannelsOwnershipVerticalList
          channels={channels}
          handleRemoveChannel={handleRemoveChannel}
          handleOwnershipSelect={handleChannelOwnershipSelect}
          handleVerticalSelect={handleChannelVerticalSelect}
        />
      )}
    </>
  );
}

function ChannelsDate({ channels, handleRemoveChannel, handleDate }) {
  return (
    <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap} margin={{ bottom: 2 }}>
      <Layout className="tw-form-group__label-container" padding={{ right: 2 }} flexShrink={0}></Layout>
      <Layout flexGrow={1} position={Position.Relative}>
        <Layout margin={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center}>
          <Layout flexGrow={1}>
            <div style={{ width: 210 }}>
              <CoreText fontWeight={FontWeight.Bold}>Added Channels</CoreText>
            </div>
          </Layout>
          <Layout flexGrow={1} fullWidth>
            <CoreText fontWeight={FontWeight.Bold}>Custom Start &amp; End Date</CoreText>
          </Layout>
        </Layout>
        <Layout display={Display.Flex} alignContent={AlignContent.Start} flexWrap={FlexWrap.Wrap}>
          {channels.length === 0 ? (
            <Layout display={Display.Block}>
              <NoResults results="Channels" />
              <CoreText>Search and Select a Channel</CoreText>
            </Layout>
          ) : (
            channels.map(channel => (
              <ChannelsDateItem key={channel.channel_id} channel={channel} handleRemoveChannel={handleRemoveChannel} handleDate={handleDate} />
            ))
          )}
        </Layout>
        <Layout margin={{ top: 0.5 }}>
          <CoreText fontSize={FontSize.Size7} color={Color.Alt2}>
            All channels included in this contract and their custom start/end dates, if applicable. Please leave custom dates blank if not applicable.
            channel
          </CoreText>
        </Layout>
      </Layout>
    </Layout>
  );
}

function ChannelsDateItem({ channel, handleRemoveChannel, handleDate }) {
  const { channel_id, channel_login, profile_image, custom_start_date, custom_end_date } = channel;
  const [startDate, setStartDate] = useState(custom_start_date ? new Date(getMomentDate(custom_start_date)) : null);
  const [endDate, setEndDate] = useState(custom_end_date ? new Date(getMomentDate(custom_end_date)) : null);

  const updateStartDate = e => {
    setStartDate(e);
    handleDate(e, channel_id, endDate);
  };

  const updateEndDate = e => {
    setEndDate(e);
    handleDate(startDate, channel_id, e);
  };

  return (
    <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ bottom: 0.5 }}>
      <Layout flexGrow={1} fullWidth padding={{ right: 1 }}>
        <div style={{ height: 30, width: 200 }}>
          <StyledLayout
            background={Background.Alt2}
            display={Display.Flex}
            alignItems={AlignItems.Center}
            borderRadius={BorderRadius.Medium}
            fullHeight
            fullWidth>
            <Layout margin={0.5}>
              <Avatar size={24} src={profile_image != null ? profile_image : '/images/404-avatar.png'} alt="" userLogin={channel_login} />
            </Layout>
            <CoreText fontSize={FontSize.Size6} ellipsis>
              {channel_login}
            </CoreText>
          </StyledLayout>
        </div>
      </Layout>
      <Layout flexGrow={1} fullWidth>
        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
          <div style={{ height: 30, width: 235 }}>
            <StyledLayout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center} fullHeight fullWidth>
              <Layout margin={{ right: 0.5 }} fullWidth>
                <DatePicker defaultDate={startDate} onChange={e => updateStartDate(e)} maxDate={endDate} />
              </Layout>
              <Layout fullWidth>
                <DatePicker defaultDate={endDate} onChange={e => updateEndDate(e)} minDate={startDate} />
              </Layout>
            </StyledLayout>
          </div>
          <Layout margin={{ left: 0.5 }}>
            <ButtonIcon onClick={e => handleRemoveChannel(e, channel_id)} icon={SVGAsset.Close} />
          </Layout>
        </Layout>
      </Layout>
    </Layout>
  );
}

function ChannelsType({ channels, handleRemoveChannel, handleTypeSelect }) {
  return (
    <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap} margin={{ bottom: 2 }}>
      <Layout className="tw-form-group__label-container" padding={{ right: 2 }} flexShrink={0}></Layout>

      <Layout flexGrow={1} position={Position.Relative}>
        <Layout margin={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center}>
          <Layout flexGrow={1} fullWidth>
            <CoreText fontWeight={FontWeight.Bold}>Connected Channels</CoreText>
          </Layout>
          <Layout flexGrow={1} fullWidth>
            <CoreText fontWeight={FontWeight.Bold}>Channel Type</CoreText>
          </Layout>
        </Layout>
        <SimpleBarReact style={{ maxHeight: 300 }}>
          <Layout display={Display.Flex} alignContent={AlignContent.Start} flexWrap={FlexWrap.Wrap}>
            {channels.length === 0 ? (
              <Layout display={Display.Block}>
                <NoResults results="Channels" />
                <CoreText>Search and Select a Channel</CoreText>
              </Layout>
            ) : (
              channels.map(channel => {
                const { channel_id, channel_login, profile_image } = channel;
                return (
                  <Layout key={channel_id} display={Display.Flex} alignItems={AlignItems.Center} margin={{ bottom: 0.5 }}>
                    <Layout flexGrow={1} fullWidth padding={{ right: 1 }}>
                      <div style={{ height: 30, width: 280 }}>
                        <StyledLayout
                          background={Background.Alt2}
                          display={Display.Flex}
                          alignItems={AlignItems.Center}
                          borderRadius={BorderRadius.Medium}
                          fullHeight
                          fullWidth>
                          <Layout margin={0.5}>
                            <Avatar size={24} src={profile_image != null ? profile_image : '/images/404-avatar.png'} alt="" userLogin={channel_login} />
                          </Layout>
                          <CoreText fontSize={FontSize.Size6} ellipsis>
                            {channel_login}
                          </CoreText>
                        </StyledLayout>
                      </div>
                    </Layout>
                    <Layout flexGrow={1} fullWidth>
                      <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                        <div style={{ height: 30, width: 180 }}>
                          <Select defaultValue={(channel && channel.channel_type) || '0'} onChange={e => handleTypeSelect(e.currentTarget.value, channel_id)}>
                            <option disabled value="">
                              Select Channel Type
                            </option>
                            {channelTypeOptions.map(type => {
                              return (
                                <option key={type} value={type}>
                                  {type}
                                </option>
                              );
                            })}
                          </Select>
                        </div>
                        <Layout margin={{ left: 0.5 }}>
                          <ButtonIcon onClick={e => handleRemoveChannel(e, channel_id)} icon={SVGAsset.Close} />
                        </Layout>
                      </Layout>
                    </Layout>
                  </Layout>
                );
              })
            )}
          </Layout>
        </SimpleBarReact>
        <Layout margin={{ top: 0.5 }}>
          <CoreText fontSize={FontSize.Size7} color={Color.Alt2}>
            All channels included in this event and their channel type.
          </CoreText>
        </Layout>
      </Layout>
    </Layout>
  );
}

function ChannelsOwnershipVerticalList({ channels, handleRemoveChannel, handleOwnershipSelect, handleVerticalSelect }) {
  return (
    <Layout margin={{ bottom: 2 }}>
      <Layout margin={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center}>
        <div style={{ width: 185 }}>
          <CoreText fontWeight={FontWeight.Bold}>Connected Channels</CoreText>
        </div>
        <div style={{ width: 140 }}>
          <CoreText fontWeight={FontWeight.Bold}>
            <LabelRequired />
            Channel Ownership
          </CoreText>
        </div>
        <div>
          <CoreText fontWeight={FontWeight.Bold}>Channel Vertical</CoreText>
        </div>
      </Layout>
      <SimpleBarReact style={{ maxHeight: 300 }}>
        <Layout display={Display.Flex} alignContent={AlignContent.Start} flexWrap={FlexWrap.Wrap}>
          {channels.length === 0 ? (
            <Layout display={Display.Block}>
              <NoResults results="Channels" />
              <CoreText>Search and Select a Channel</CoreText>
            </Layout>
          ) : (
            channels.map((channel, i) => (
              <ChannelPCCItem
                key={channel.channel_id}
                channel={channel}
                i={i}
                handleRemoveChannel={handleRemoveChannel}
                handleOwnershipSelect={handleOwnershipSelect}
                handleVerticalSelect={handleVerticalSelect}
              />
            ))
          )}
        </Layout>
      </SimpleBarReact>
      <Layout margin={{ top: 0.5 }}>
        <CoreText fontSize={FontSize.Size7} color={Color.Alt2}>
          All channels included with this content creator
        </CoreText>
      </Layout>
    </Layout>
  );
}

function ChannelPCCItem({ channel, i, handleRemoveChannel, handleOwnershipSelect, handleVerticalSelect }) {
  const { channel_id, channel_login, profile_image, channel_ownership, channel_vertical } = channel;

  return (
    <Layout key={channel_id} display={Display.Flex} alignItems={AlignItems.Center} margin={{ bottom: 0.5 }}>
      <Layout flexGrow={1} fullWidth padding={{ right: 0.5 }}>
        <div style={{ height: 30, width: 180 }}>
          <StyledLayout
            background={Background.Alt2}
            display={Display.Flex}
            alignItems={AlignItems.Center}
            borderRadius={BorderRadius.Medium}
            fullHeight
            fullWidth>
            <Layout margin={0.5}>
              <Avatar size={24} src={profile_image != null ? profile_image : '/images/404-avatar.png'} alt="" userLogin={channel_login} />
            </Layout>
            <CoreText fontSize={FontSize.Size6} ellipsis>
              {channel_login}
            </CoreText>
          </StyledLayout>
        </div>
      </Layout>
      <Layout flexGrow={1} fullWidth>
        <Layout display={Display.Flex} alignItems={AlignItems.Center}>
          <div style={{ height: 30, width: 280 }}>
            <StyledLayout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center} fullHeight fullWidth>
              <Layout margin={{ right: 0.5 }} fullWidth>
                <Select value={channel_ownership || channelOwnershipOptions[0]} onChange={e => handleOwnershipSelect(e.currentTarget.value, channel_id)} required>
                  <option disabled value="">
                    Select Ownership
                  </option>
                  {channelOwnershipOptions.map(ownership => {
                    return (
                      <option key={ownership} value={ownership}>
                        {ownership}
                      </option>
                    );
                  })}
                </Select>
              </Layout>
              <Layout fullWidth margin={{ right: 0.5 }}>
                <Select value={channel_vertical || ''} onChange={e => handleVerticalSelect(e.currentTarget.value, channel_id)}>
                  <option disabled value="">
                    Select Vertical
                  </option>
                  {channelVerticalOptions.map(vertical => {
                    return (
                      <option key={vertical} value={vertical}>
                        {vertical}
                      </option>
                    );
                  })}
                </Select>
              </Layout>
            </StyledLayout>
          </div>
          <ButtonIcon onClick={e => handleRemoveChannel(e, channel_id)} icon={SVGAsset.Close} />
        </Layout>
      </Layout>
    </Layout>
  );
}

export default FormChannelsSelect;
