import moment from 'moment';
import React from 'react';
import { Color, CoreText, Input, InputType, Layout, SVGAsset } from 'twitch-core-ui';
import { useEffectOnce } from '../utils/useEffectOnce';

// eg 1:30, 16:40, etc
const TIME_FORMAT_24_HRS = "HH:mm";

// eg 1:30 am, 4:40 pm
const TIME_FORMAT_AMPM = "hh:mm A";

// eg 1:30 am, 4:40 pm for display
const TIME_FORMAT_SHORT_TIME = "LT";

function FormInputTime({ time, setTime, isInvalidTime, setIsInvalidTime, disabled = false, required = false }) {
  const [displayTime, setDisplayTime] = React.useState<string>("");
  const [focused, setFocused] = React.useState<boolean>(false);

  // This initializes the textbox to either a valid HH:mm, or to nothing.
  let defaultTime: string | undefined = undefined;
  if (time) {
    let defaultTimeMoment = moment(time, TIME_FORMAT_24_HRS);
    if (defaultTimeMoment.isValid()) {
      defaultTime = defaultTimeMoment.format(TIME_FORMAT_24_HRS);
    }
  }

  useEffectOnce(() => {
    formatAsTime(defaultTime);
  });

  const onFocus = () => {
    setFocused(true);
  }


  const onBlur = () => {
    setFocused(false);
  }

  const formatAsTime = (value?: string) => {
    if (!value) {
      setIsInvalidTime(true);
      return;
    }

    value = value.trim();

    // Try am/pm or military time
    let formatted = moment(value, TIME_FORMAT_AMPM);
    if (!formatted.isValid()) {
      formatted = moment(value, TIME_FORMAT_24_HRS);
      if (!formatted.isValid()) {
        setIsInvalidTime(true);
        return;
      }
    }

    var displayString = formatted.format(TIME_FORMAT_SHORT_TIME);
    var formattedString = formatted.format(TIME_FORMAT_24_HRS);
    setDisplayTime(displayString);
    setTime(formattedString);
    setIsInvalidTime(false);
  }

  const onChange = e => {
    let { target } = e;
    let { value } = target;

    formatAsTime(value);
  };

  const hintValue = isInvalidTime ? "Invalid" : displayTime;

  return (
    <Layout>
    <Input
      type={InputType.Text}
      placeholder={TIME_FORMAT_24_HRS}
      defaultValue={defaultTime}
      onChange={onChange}
      onKeyDown={onChange}
      onFocus={onFocus}
      onBlur={onBlur}
      iconRight
      icon={SVGAsset.Timeout}
      error={isInvalidTime}
      maxLength={8}
      disabled={disabled}
      required={required}
    />
    {focused && <CoreText color={Color.Alt2}>({hintValue})</CoreText>}
    </Layout>
  );
}

export default FormInputTime;
