import React, { useState, useEffect } from 'react';
import SimpleBarReact from 'simplebar-react';
import Cookies from 'js-cookie';
import {
  Display,
  Layout,
  CoreText,
  SVG,
  SVGAsset,
  SVGType,
  TextType,
  JustifyContent,
  AlignItems,
  AlertBanner,
  AlertBannerType,
  FontSize,
  Position,
  DropDownMenuItem,
  FontWeight,
  Color,
  SearchInput,
  InputSize,
  StyledLayout,
  BorderRadius,
  Button,
  Accordion,
  ButtonSize,
  Pill
} from 'twitch-core-ui';
import Loading from './loading';
import NoResults from './no-results';
import { useFetchPOST } from '../hooks/useFetch';
import { ATLAS_BASE_URL } from '../utils/config';
import { writeStorage } from '@rehooks/local-storage';
import { timezoneOptions } from '../utils/options';
import { getDefaultTimezoneOption } from '../utils/dateTimeHelpers';
import './styles/PccWrapper.scss';

function GettingStarted() {
  const [searchQuery, setSearchQuery] = useState('');
  const [productOwner, setProductOwner] = useState(localStorage.getItem('productOwner') || '');
  const [timezone, setTimezone] = useState('');
  const [data, loading, error] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-managed-creators`);
  const [isPccSelected, setIsPccSelected] = useState(localStorage.getItem('productOwner') === null ? false : localStorage.getItem('productOwner').length);
  const [openIndex, setOpenIndex] = useState(0);

  useEffect(() => {
    if (localStorage.getItem('timezone') === null) {
      localStorage.setItem('timezone', JSON.stringify(getDefaultTimezoneOption()));
    }
    if (localStorage.getItem('timezone') !== null) {
      setTimezone(JSON.parse(localStorage.getItem('timezone')).value);
    }
  }, []);

  const selectProductOwner = (pccId, pccName) => {
    if (productOwner === pccName) {
      return;
    } else {
      setProductOwner(pccName);
      writeStorage('productOwner', pccName);
      writeStorage('productOwnerID', pccId);
      setIsPccSelected(true);
      setOpenIndex(1);
    }
  };

  const selectTimezone = tzObj => {
    if (timezone === tzObj) {
      return;
    } else {
      setTimezone(tzObj.value);
      writeStorage('timezone', tzObj);
      setOpenIndex(null);
    }
  };

  return (
    <Layout>
      {error && <AlertBanner type={AlertBannerType.Error} status="Error" message={error.toString()} />}
      <Layout padding={5}>
        <Layout display={Display.Flex} justifyContent={JustifyContent.Center} alignItems={AlignItems.Center} fullWidth>
          <SVG asset={SVGAsset.LogoGlitch} width={60} height={45} type={SVGType.Brand} />
          <CoreText type={TextType.H2}>Welcome to Atlas{`, ${Cookies.get('profile')}` || ''}!</CoreText>
        </Layout>
        <Layout display={Display.Flex} justifyContent={JustifyContent.Center} alignItems={AlignItems.Center} fullWidth>
          <StyledLayout className="pcc-wrapper" margin={{ top: 2 }} padding={2} border borderRadius={BorderRadius.Medium}>
            <Layout margin={{ bottom: 2 }} display={Display.Flex} alignItems={AlignItems.Center}>
              <Layout margin={{ right: 0.5 }}>
                <SVG asset={SVGAsset.Rocket} type={SVGType.Alt2} />
              </Layout>
              <CoreText type={TextType.H3}>Getting Started</CoreText>
            </Layout>
            <Accordion
              initialOpenIndex={openIndex}
              items={[
                {
                  header: {
                    title: isPccSelected ? 'Premium Content Creator Selected' : 'Select a Premium Content Creator',
                    description: isPccSelected ? productOwner : null,
                    imageProps: {
                      src: '/images/getting-started-pcc.png',
                      alt: 'Premium Content Creator Image'
                    },
                    isOpen: openIndex === 0
                  },
                  body: {
                    children: (
                      <Layout fullWidth>
                        <Layout margin={{ bottom: 1 }}>
                          <SearchInput
                            placeholder="Search Premium Content Creators"
                            value={searchQuery}
                            size={InputSize.Large}
                            onChange={e => setSearchQuery(e.currentTarget.value)}
                          />
                        </Layout>
                        <SimpleBarReact style={{ maxHeight: 250 }}>
                          {loading ? (
                            <Layout padding={{ x: 1 }}>
                              <Loading />
                            </Layout>
                          ) : !!data.data.viewable_creators ? (
                            data.data.viewable_creators
                              .filter(contentCreator => contentCreator.premium_content_creator_name.toLowerCase().includes(searchQuery.toLowerCase()))
                              .map((contentCreator, i) => {
                                const { premium_content_creator_id, premium_content_creator_name } = contentCreator;
                                return (
                                  <Layout key={i} className="content-creator--container" position={Position.Relative}>
                                    <Layout className="content-creator--option">
                                      <DropDownMenuItem onClick={() => selectProductOwner(premium_content_creator_id, premium_content_creator_name)}>
                                        <Layout display={Display.Flex} alignItems={AlignItems.Center} padding={{ left: 0.5, right: 4 }}>
                                          <CoreText
                                            color={premium_content_creator_name === productOwner ? Color.Link : Color.Base}
                                            fontSize={FontSize.Size4}
                                            fontWeight={FontWeight.Bold}>
                                            {premium_content_creator_name}
                                          </CoreText>
                                        </Layout>
                                      </DropDownMenuItem>
                                    </Layout>
                                  </Layout>
                                );
                              })
                          ) : (
                            <NoResults results="channels" />
                          )}
                        </SimpleBarReact>
                      </Layout>
                    )
                  }
                },
                {
                  header: {
                    title: timezone.length ? 'Time Zone Selected' : 'Select Time Zone',
                    description: timezone.length ? timezone : null,
                    imageProps: {
                      src: '/images/getting-started-tz.png',
                      alt: 'Time Zone Image'
                    },
                    isOpen: openIndex === 1
                  },
                  body: {
                    children: (
                      <Layout fullWidth>
                        <SimpleBarReact style={{ maxHeight: 250 }}>
                          {timezoneOptions.map(option => (
                            <DropDownMenuItem
                              key={option && option.value}
                              onClick={() => selectTimezone(option)}
                              selected={option && option.value === timezone}>
                              <CoreText>{option.name}</CoreText>
                              {option && option.offset === getDefaultTimezoneOption().offset && <Pill label="Local" />}
                            </DropDownMenuItem>
                          ))}
                        </SimpleBarReact>
                      </Layout>
                    )
                  }
                }
              ]}
            />
            <Layout margin={{ top: 2 }}>
              <Button size={ButtonSize.Large} disabled={!isPccSelected} fullWidth linkTo="/">
                View Atlas Dashboard
              </Button>
            </Layout>
          </StyledLayout>
        </Layout>
      </Layout>
    </Layout>
  );
}

export default GettingStarted;
