import React from 'react';
import Cookies from 'js-cookie';
import SimpleBarReact from 'simplebar-react';
import { Display, AlignItems, FontSize, Color, Layout, StyledLayout, Position, ResponsiveWrapper } from 'twitch-core-ui';
import NavHeader from './nav-header';
import Sidebar from './sidebar';
import { ATLAS_BASE_URL, NAV_HEIGHT } from '../utils/config';
import withWindowDimensions from './with-window-dimensions';
import { PermissionsProvider } from '../context/permissions-context';
import { MasqueradeProvider } from '../context/masquerade-context';
import { fetchWithTimeout } from '../utils/helpers';
import './styles/MainWrapper.scss';
import { DataContextProvider } from '../context/data-context';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';
import { GQLContextProvider } from '../context/gql-context';

interface Props {
  children: React.ReactChild;
  windowHeight: number;
}

interface OwnedChannelGrouping {
  product_name: string;
  channels: OwnedChannel[];
}

interface OwnedChannel {
  channel_login: string;
  channel_status: string;
  channel_profile_image: string;
  channel_category: string;
  channel_id: Number;
  viewers: Number;
}

interface State {
  groupings: OwnedChannelGrouping[];
  loading: boolean;
  filteredGroupings: OwnedChannelGrouping[];
  isCollapsed: boolean;
}

class MainWrapper extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);
    this.state = {
      groupings: new Array<OwnedChannelGrouping>(),
      loading: true,
      filteredGroupings: new Array<OwnedChannelGrouping>(),
      isCollapsed: localStorage.getItem('isSidebarCollapsed') === 'true'
    };
  }

  componentDidMount() {
    this.getChannels();
  }

  public getChannels() {
    this.updateGroupings();
  }

  private updateGroupings = () => {
    const token = Cookies.get('token');
    let options = {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        Authorization: `Bearer ${token}`
      }
    };
    const contentCreatorID = localStorage.getItem(PCCID_LOCAL_STORAGE_KEY);

    fetchWithTimeout(`${ATLAS_BASE_URL}/event/get-managed-channels/${contentCreatorID}`, options)
      .then(res => res.json())
      .then(data => {
        console.log(`/get-managed-channels:`, data);
        if (data.data !== null) {
          const { grouped_channels } = data.data && data.data;
          this.setState({ filteredGroupings: grouped_channels, groupings: grouped_channels, loading: false });
        } else {
          this.setState({ filteredGroupings: [], groupings: [], loading: false });
        }
      })
      .catch(error => {
        console.error(`/get-managed-channels:`, error);
        this.setState({ filteredGroupings: [], groupings: [], loading: false });
      });
  };

  private collapseSidebar = () => {
    this.setState(
      prevState => ({
        isCollapsed: !prevState.isCollapsed
      }),
      () => localStorage.setItem('isSidebarCollapsed', this.state.isCollapsed.toString())
    );
  };

  public render() {
    const { children, windowHeight } = this.props;
    const { filteredGroupings, groupings, loading, isCollapsed } = this.state;
    const contentCreators = groupings == null ? [] : groupings;
    const filteredContentCreators = filteredGroupings == null ? [] : filteredGroupings;

    return (
      <Layout className={`main-wrapper ${isCollapsed ? 'collapsed' : ''}`} display={Display.Flex} alignItems={AlignItems.Center} fullWidth>
        <PermissionsProvider>
          <MasqueradeProvider>
            <GQLContextProvider>
              <DataContextProvider>
                <NavHeader loading={loading} groupings={contentCreators} updateGroupings={this.updateGroupings} />
                <Sidebar loading={loading} groupings={filteredContentCreators} isCollapsed={isCollapsed} setIsCollapsed={this.collapseSidebar} />
                <StyledLayout
                  className={`main-wrapper--body ${isCollapsed ? 'sidebar--collapsed' : ''}`}
                  fontSize={FontSize.Size4}
                  color={Color.Base}
                  fullWidth
                  position={Position.Relative}>
                  <SimpleBarReact style={{ maxHeight: windowHeight - NAV_HEIGHT }}>
                    <Layout padding={3}>
                      <div style={{ minHeight: 450 }}>
                        <ResponsiveWrapper>{children}</ResponsiveWrapper>
                      </div>
                    </Layout>
                  </SimpleBarReact>
                </StyledLayout>
              </DataContextProvider>
            </GQLContextProvider>
          </MasqueradeProvider>
        </PermissionsProvider>
      </Layout>
    );
  }
}

export default withWindowDimensions(MainWrapper);
