import React, { useState } from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  Layout,
  SVGAsset,
  CoreText,
  TextType,
  Display,
  Position,
  StyledLayout,
  FlexDirection,
  Background,
  Color,
  ZIndex,
  ButtonIcon,
  BorderRadius,
  ButtonIconSize,
  AlignItems,
  JustifyContent,
  TextTransform,
  Placeholder,
  Tooltip,
  TooltipDirection,
  Overflow
} from 'twitch-core-ui';
import { textify } from '../utils/helpers';
import './styles/MetricsConsole.scss';

function MetricsConsole({ metrics, loading, name }) {
  const { child_metrics, elapsed_ms, metric_name } = metrics;
  const [isVisible, setIsVisible] = useState(localStorage.getItem('isMetricsVisible') === 'true');

  const generateHeatmapClassNames = (ms, slow_ms = 1000, fast_ms = 300) => {
    let heatmapClasses = 'metric--ms ';
    heatmapClasses += ms > slow_ms ? 'slow' : ms < fast_ms ? 'fast' : 'medium';
    return heatmapClasses;
  };

  const showMetrics = () => {
    setIsVisible(true);
    localStorage.setItem('isMetricsVisible', true.toString());
  };

  const hideMetrics = () => {
    setIsVisible(false);
    localStorage.setItem('isMetricsVisible', false.toString());
  };

  return (
    <StyledLayout
      className="metrics-console"
      position={Position.Fixed}
      display={Display.Flex}
      background={Background.Base}
      color={Color.Base}
      border
      borderRadius={BorderRadius.Large}
      overflow={Overflow.Hidden}
      zIndex={ZIndex.Above}
      attachLeft
      attachBottom
      margin={1}>
      {!isVisible ? (
        <StyledLayout background={Background.Accent} fullWidth>
          <Tooltip label="Metrics" direction={TooltipDirection.Right}>
            <ButtonIcon icon={SVGAsset.OP} onClick={showMetrics} overlay />
          </Tooltip>
        </StyledLayout>
      ) : (
        <StyledLayout display={Display.Flex} fullWidth>
          <StyledLayout
            className="metrics-console--siderbar"
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            flexGrow={0}
            flexShrink={1}
            borderRight>
            <StyledLayout background={Background.Accent}>
              <ButtonIcon icon={SVGAsset.OP} size={ButtonIconSize.Large} overlay onClick={hideMetrics} />
            </StyledLayout>
          </StyledLayout>
          <Layout className="metrics-console--container">
            <StyledLayout
              className="metrics-console--header"
              borderBottom
              padding={{ left: 1, right: 0.5 }}
              display={Display.Flex}
              alignItems={AlignItems.Center}
              justifyContent={JustifyContent.Between}>
              <CoreText type={TextType.H4}>{name} Metrics</CoreText>
              <ButtonIcon icon={SVGAsset.Close} onClick={hideMetrics} />
            </StyledLayout>
            <SimpleBarReact style={{ maxHeight: 213 }}>
              <Layout className="metrics-console--body" margin={1}>
                {loading ? (
                  <Layout>
                    <Layout padding={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between}>
                      <Placeholder height={17.1} />
                    </Layout>
                    <Layout padding={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between}>
                      <Placeholder height={17.1} />
                    </Layout>
                    <Layout padding={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between}>
                      <Placeholder height={17.1} />
                    </Layout>
                    <Layout padding={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between}>
                      <Placeholder height={17.1} />
                    </Layout>
                  </Layout>
                ) : (
                  <>
                    <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between}>
                      <CoreText type={TextType.H6} transform={TextTransform.Capitalize}>
                        {metric_name && textify(metric_name)}
                      </CoreText>
                      <Layout className={generateHeatmapClassNames(elapsed_ms, 1500, 500)}>
                        <CoreText type={TextType.H6}>{elapsed_ms && elapsed_ms.toLocaleString()} ms</CoreText>
                      </Layout>
                    </Layout>
                    <ul>
                      {child_metrics &&
                        child_metrics.map((metric, i) => {
                          const { elapsed_ms, metric_name } = metric;
                          return (
                            <li key={i}>
                              <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between}>
                                <CoreText type={TextType.H6} transform={TextTransform.Capitalize}>
                                  {metric_name && textify(metric_name)}
                                </CoreText>
                                <Layout className={generateHeatmapClassNames(elapsed_ms)}>
                                  <CoreText type={TextType.H6}>{elapsed_ms && elapsed_ms.toLocaleString()} ms</CoreText>
                                </Layout>
                              </Layout>
                            </li>
                          );
                        })}
                    </ul>
                  </>
                )}
              </Layout>
            </SimpleBarReact>
          </Layout>
        </StyledLayout>
      )}
    </StyledLayout>
  );
}

export default MetricsConsole;
