import React, { useState } from 'react';
import SimpleBarReact from 'simplebar-react';
import {
  CoreText,
  Layout,
  Display,
  AlignItems,
  Position,
  DropDownMenu,
  DropDownMenuItem,
  BalloonSize,
  Transition,
  TransitionType,
  ZIndex,
  FlexWrap,
  Tag,
  TagAction,
  SearchInput
} from 'twitch-core-ui';
import ClickOutDetector from './click-out-detector';
import NoResults from './no-results';
import './styles/Form.scss';

function MultiselectAccountManager({ accountManagers, setAccountManagers, dropdownOptions }) {
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');

  const toggleDropdown = e => {
    e.preventDefault();
    setIsDropdownOpen(!isDropdownOpen);
  };

  const selectOption = (e, am) => {
    e.preventDefault();
    setAccountManagers([...accountManagers, am]);
  };

  const deleteOption = (e, am) => {
    e.preventDefault();
    setAccountManagers(accountManagers.filter(o => o.account_manager_id !== am.account_manager_id));
  };

  const filteredAccountManagers =
    dropdownOptions &&
    dropdownOptions
      .filter(am => !accountManagers.includes(am))
      .filter(am => {
        const { account_manager_first_name, account_manager_last_name, account_manager_ldap_name } = am;
        return (
          account_manager_first_name.toLowerCase().includes(searchQuery.toLowerCase()) ||
          account_manager_last_name.toLowerCase().includes(searchQuery.toLowerCase()) ||
          account_manager_ldap_name.toLowerCase().includes(searchQuery.toLowerCase())
        );
      });

  return (
    <>
      <Layout
        display={accountManagers && accountManagers.length ? Display.Flex : Display.Hide}
        flexWrap={FlexWrap.Wrap}
        alignItems={AlignItems.Start}
        fullWidth>
        {accountManagers &&
          accountManagers.map(am => (
            <Layout key={am.account_manager_id} margin={{ bottom: 0.5, right: 0.5 }} ellipsis>
              <Tag label={am.account_manager_ldap_name} action={TagAction.Remove} onClick={e => deleteOption(e, am)} />
            </Layout>
          ))}
      </Layout>
      <ClickOutDetector onClickOut={() => setIsDropdownOpen(false)}>
        <SearchInput placeholder="Add Account Manager(s)" value={searchQuery} onChange={e => setSearchQuery(e.currentTarget.value)} onFocus={toggleDropdown} />
        <Layout position={Position.Relative} fullWidth zIndex={ZIndex.Above}>
          <Transition type={TransitionType.SlideOverTop} show={isDropdownOpen}>
            <DropDownMenu show={isDropdownOpen} size={BalloonSize.Auto}>
              <SimpleBarReact style={{ maxHeight: 200, width: 470 }}>
                {filteredAccountManagers && filteredAccountManagers.length ? (
                  filteredAccountManagers.map(am => {
                    const { account_manager_id, account_manager_first_name, account_manager_last_name, account_manager_ldap_name } = am;
                    return (
                      <DropDownMenuItem key={account_manager_id} onClick={e => selectOption(e, am)}>
                        <CoreText>
                          {account_manager_first_name} {account_manager_last_name} ({account_manager_ldap_name})
                        </CoreText>
                      </DropDownMenuItem>
                    );
                  })
                ) : (
                  <NoResults results="account managers" small />
                )}
              </SimpleBarReact>
            </DropDownMenu>
          </Transition>
        </Layout>
      </ClickOutDetector>
    </>
  );
}

export default MultiselectAccountManager;
