import React from 'react';
import SimpleBarReact from 'simplebar-react';
import Cookies from 'js-cookie';
import {
  Display,
  FontSize,
  AlignItems,
  Layout,
  Position,
  Button,
  ButtonType,
  Transition,
  TransitionType,
  DropDownMenu,
  DropDownMenuItem,
  CoreText,
  DropDownMenuHeading,
  FontWeight,
  BalloonSize,
  SVGAsset,
  JustifyContent,
  ButtonSize,
  SVG,
  Color,
  SearchInput,
  Overflow,
  StyledLayout,
  ButtonIcon,
  ButtonIconType
} from 'twitch-core-ui';
import Loading from './loading';
import NoResults from './no-results';
import ContentCreatorModal from './content-creator-modal';
import ClickOutDetector from './click-out-detector';
import { ATLAS_BASE_URL } from '../utils/config';
import { writeStorage } from '@rehooks/local-storage';
import { fetchWithTimeout } from '../utils/helpers';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

interface Props {
  updateGroupings: any;
  canEdit: boolean;
}

interface State {
  isDropdownVisible: boolean;
  productOwner: string;
  productOwnerId: number;
  isModalOpen: boolean;
  modalTitle: string;
  selectedContentCreator: ContentCreator;
  loading: boolean;
  ownedContentCreators: ContentCreator[];
  viewableContentCreators: ContentCreator[];
  searchQuery: string;
  isDevMode: boolean;
}

interface ContentCreator {
  premium_content_creator_id?: number;
  premium_content_creator_name: string;
  premium_content_creator_type: string;
  pcc_am_map_id: string;
}

class NavContentCreator extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);
    this.state = {
      isDropdownVisible: false,
      productOwner: '',
      productOwnerId: null,
      isModalOpen: false,
      modalTitle: 'Add Premium Content Creator',
      selectedContentCreator: {} as ContentCreator,
      loading: true,
      ownedContentCreators: [],
      viewableContentCreators: [],
      searchQuery: '',
      isDevMode: window.location.hostname === 'localhost'
    };
  }

  componentDidMount() {
    this.setState({
      productOwner: localStorage.getItem('productOwner') || '',
      productOwnerId: parseInt(localStorage.getItem(PCCID_LOCAL_STORAGE_KEY)) || 0
    });
    this.loadCreators();
  }

  private loadCreators() {
    let token = Cookies.get('token');

    fetchWithTimeout(`${ATLAS_BASE_URL}/event/get-managed-creators`, {
      method: 'POST',
      headers: {
        Accept: 'application/json',
        'Content-Type': 'application/json',
        Authorization: `Bearer ${token}`
      }
    })
      .then(response => response.json())
      .then(data => {
        console.log(`/get-managed-creators:`, data);
        if (data.success) {
          const creatorData = data && data.data && data.data.viewable_creators;
          this.setState({
            loading: false,
            ownedContentCreators: data.data.owned_creators,
            viewableContentCreators: data.data.viewable_creators
          });
          if (this.state.productOwner === '') {
            this.selectProductOwner(creatorData[0].premium_content_creator_id, creatorData[0].premium_content_creator_name);
          } else {
          }
        } else {
          this.setState({
            loading: false,
            ownedContentCreators: [],
            viewableContentCreators: []
          });
        }
        if (this.state.viewableContentCreators.length === 0 && !this.state.isDevMode) {
          window.location.href = '/no-pcc';
        }
      })
      .catch(err => {
        console.error(`/get-managed-creators:`, err);
        this.setState({
          loading: false,
          ownedContentCreators: [],
          viewableContentCreators: []
        });
        if (this.state.viewableContentCreators.length === 0 && !this.state.isDevMode) {
          window.location.href = '/no-pcc';
        }
      });
  }

  private closeDropdown() {
    this.setState({ isDropdownVisible: false });
  }

  private selectProductOwner = (productOwnerId, productOwner) => {
    this.closeDropdown();
    if (this.state.productOwner === productOwner) {
      return;
    } else {
      this.setState({ productOwner: productOwner });
      this.setState({ productOwnerId: productOwnerId });
      writeStorage('productOwner', productOwner);
      writeStorage('productOwnerID', productOwnerId);
      this.props.updateGroupings(productOwner);
    }
  };

  private openContentCreatorModal = () => {
    this.setState({ isDropdownVisible: false, isModalOpen: true, modalTitle: 'Add Premium Content Creator' });
  };

  private editContentCreator = (e, contentCreator) => {
    e.preventDefault();
    this.openContentCreatorModal();
    this.setState({ modalTitle: 'Edit Premium Content Creator' });

    fetchWithTimeout(`${ATLAS_BASE_URL}/event/get-premium-content-creator-list`, {
      method: 'POST',
      headers: {
        Accept: 'application/json',
        'Content-Type': 'application/json',
        Authorization: `Bearer ${Cookies.get('token')}`
      },
      body: JSON.stringify({
        eventtype: 6,
        'req-cols': ['*'],
        filters: [
          {
            'filter-type': 1,
            key: 'premium_content_creator_id',
            v1: contentCreator
          }
        ]
      })
    })
      .then(response => response.json())
      .then(data => {
        console.log(`/get-premium-content-creator-list:`, data);
        this.setState({ selectedContentCreator: data.data.items[0] });
      })
      .catch(err => {
        console.error(`/get-premium-content-creator-list:`, err);
        this.setState({
          loading: false,
          viewableContentCreators: []
        });
      });
  };

  public render() {
    const {
      isDropdownVisible,
      productOwner,
      isModalOpen,
      modalTitle,
      selectedContentCreator,
      loading,
      ownedContentCreators,
      viewableContentCreators,
      searchQuery,
      productOwnerId
    } = this.state;

    const filteredViewableContentCreators =
      viewableContentCreators &&
      viewableContentCreators
        .filter(contentCreator => JSON.stringify(ownedContentCreators).indexOf(JSON.stringify(contentCreator)) < 0)
        .filter(contentCreator => contentCreator.premium_content_creator_name.toLowerCase().includes(searchQuery.toLowerCase()));

    const filteredOwnedContentCreators =
      ownedContentCreators &&
      ownedContentCreators.filter(contentCreator => contentCreator.premium_content_creator_name.toLowerCase().includes(searchQuery.toLowerCase()));

    return (
      <Layout display={Display.Flex} fullWidth alignItems={AlignItems.Center} justifyContent={JustifyContent.Center}>
        <ClickOutDetector onClickOut={() => this.setState({ isDropdownVisible: false })}>
          <Layout className="content-creator--select-btn-wrapper" position={Position.Relative}>
            <Button
              type={ButtonType.Text}
              size={ButtonSize.Large}
              onClick={() => this.setState(prevState => ({ isDropdownVisible: !prevState.isDropdownVisible }))}>
              <div style={{ width: 280 }}>
                <Layout display={Display.Flex} justifyContent={JustifyContent.Center} alignItems={AlignItems.Center}>
                  <Layout overflow={Overflow.Hidden}>
                    <CoreText fontSize={FontSize.Size4} fontWeight={FontWeight.Bold} ellipsis>
                      {productOwner || 'Select Premium Content Creator'}
                    </CoreText>
                  </Layout>
                  <SVG asset={SVGAsset.AngleDown} />
                </Layout>
              </div>
            </Button>
            {this.props.canEdit && (
              <Layout className="content-creator--edit-btn content-creator--edit-btn__header" position={Position.Absolute} attachRight margin={{ right: 0.5 }}>
                <ButtonIcon icon={SVGAsset.Edit} type={ButtonIconType.Primary} onClick={e => this.editContentCreator(e, productOwnerId)} />
              </Layout>
            )}
          </Layout>
          <Layout position={Position.Relative} fullWidth>
            <Transition type={TransitionType.SlideOverTop} show={isDropdownVisible}>
              <DropDownMenu show={isDropdownVisible} size={BalloonSize.Medium}>
                <DropDownMenuHeading>Manage Premium Content Creators</DropDownMenuHeading>
                <SearchInput placeholder="Search" value={searchQuery} onChange={e => this.setState({ searchQuery: e.currentTarget.value })} />
                <Layout margin={{ top: 2 }}>
                  <Layout margin={{ left: 1, bottom: 0.5 }}>
                    <CoreText color={Color.Alt2} italic>
                      Your assigned Premium Content Creators
                    </CoreText>
                  </Layout>
                  <SimpleBarReact style={{ maxHeight: 100 }}>
                    {loading ? (
                      <Layout padding={{ x: 1 }}>
                        <Loading />
                      </Layout>
                    ) : ownedContentCreators.length === 0 ? (
                      <NoResults results="assigned creators" />
                    ) : (
                      filteredOwnedContentCreators.map((contentCreator, i) => {
                        const { premium_content_creator_id, premium_content_creator_name } = contentCreator;
                        const isActivePCC = premium_content_creator_name === productOwner;
                        return (
                          <Layout key={i} className="content-creator--container" position={Position.Relative}>
                            <Layout className="content-creator--option">
                              <DropDownMenuItem onClick={() => this.selectProductOwner(premium_content_creator_id, premium_content_creator_name)}>
                                <Layout display={Display.Flex} alignItems={AlignItems.Center} padding={{ left: 0.5, right: 4 }}>
                                  <CoreText color={isActivePCC ? Color.Link : Color.Base} fontSize={FontSize.Size4} fontWeight={FontWeight.Bold}>
                                    {premium_content_creator_name}
                                  </CoreText>
                                </Layout>
                              </DropDownMenuItem>
                            </Layout>
                            {this.props.canEdit && (
                              <Layout className="content-creator--edit-btn" position={Position.Absolute} attachRight margin={{ right: 0.5 }}>
                                <ButtonIcon
                                  icon={SVGAsset.Edit}
                                  type={ButtonIconType.Primary}
                                  onClick={e => this.editContentCreator(e, premium_content_creator_id)}
                                />
                              </Layout>
                            )}
                          </Layout>
                        );
                      })
                    )}
                  </SimpleBarReact>
                </Layout>
                <StyledLayout borderTop padding={{ top: 2 }} margin={{ top: 1 }}>
                  <Layout margin={{ left: 1, bottom: 0.5 }}>
                    <CoreText color={Color.Alt2} italic>
                      Other Premium Content Creators
                    </CoreText>
                  </Layout>
                  <SimpleBarReact style={{ maxHeight: 300 }}>
                    {loading ? (
                      <Layout padding={{ x: 1 }}>
                        <Loading />
                      </Layout>
                    ) : filteredViewableContentCreators.length === 0 ? (
                      <NoResults results="creators" />
                    ) : (
                      filteredViewableContentCreators.map((contentCreator, i) => {
                        const { premium_content_creator_id, premium_content_creator_name } = contentCreator;
                        const isActivePCC = premium_content_creator_name === productOwner;
                        return (
                          <Layout key={i} className="content-creator--option">
                            <DropDownMenuItem onClick={() => this.selectProductOwner(premium_content_creator_id, premium_content_creator_name)}>
                              <Layout display={Display.Flex} alignItems={AlignItems.Center} padding={{ left: 0.5, right: 4 }}>
                                <CoreText color={isActivePCC ? Color.Link : Color.Base} fontSize={FontSize.Size4} fontWeight={FontWeight.Bold}>
                                  {premium_content_creator_name}
                                </CoreText>
                              </Layout>
                            </DropDownMenuItem>
                            {this.props.canEdit && (
                              <Layout className="content-creator--edit-btn" position={Position.Absolute} attachRight margin={{ right: 0.5 }}>
                                <ButtonIcon
                                  icon={SVGAsset.Edit}
                                  type={ButtonIconType.Primary}
                                  onClick={e => this.editContentCreator(e, premium_content_creator_id)}
                                />
                              </Layout>
                            )}
                          </Layout>
                        );
                      })
                    )}
                  </SimpleBarReact>
                  {this.props.canEdit && (
                    <Layout display={Display.Flex} justifyContent={JustifyContent.End} margin={{ top: 1 }}>
                      <Button onClick={this.openContentCreatorModal}>Add Premium Content Creator</Button>
                    </Layout>
                  )}
                </StyledLayout>
              </DropDownMenu>
            </Transition>
          </Layout>
        </ClickOutDetector>
        <ContentCreatorModal
          isModalOpen={isModalOpen}
          setIsModalOpen={() => this.setState(prevState => ({ isModalOpen: !prevState.isModalOpen }))}
          modalTitle={modalTitle}
          contentCreators={viewableContentCreators}
          selectedContentCreator={selectedContentCreator}
        />
      </Layout>
    );
  }
}

export default NavContentCreator;
