import React from 'react';
import Cookies from 'js-cookie';
import SimpleBarReact from 'simplebar-react';
import {
  ButtonIcon,
  SVGAsset,
  Layout,
  DropDownMenuItem,
  Position,
  FontSize,
  StyledLayout,
  Balloon,
  BalloonDirection,
  DropDownMenuSeparator,
  Overflow,
  TransitionDuration,
  TransitionGroup,
  TransitionType,
  TransitionTypeOption,
  CoreText,
  FontWeight,
  BalloonSize,
  Avatar,
  PresenceStatus,
  Display,
  AlignItems,
  FlexDirection,
  Presence,
  DropDownMenuInputItem,
  DropDownMenuInputItemType,
  JustifyContent,
  Tooltip,
  TooltipDirection,
  Pill,
  Tag
} from 'twitch-core-ui';
import ClickOutDetector from './click-out-detector';
import { getDefaultTimezoneOption } from '../utils/dateTimeHelpers';
import { timezoneOptions } from '../utils/options';

enum MenuOptions {
  Main = 'main',
  Timezone = 'timezone'
}

interface Props {
  canMasquerade: boolean;
  isMasqueradeActive: boolean;
}

interface State {
  menu: MenuOptions;
  transitionType?: TransitionTypeOption;
  isMainMenuOpen: boolean;
  isHelpMenuOpen: boolean;
  isLocal: boolean;
  currentValue: string;
  globalTimezone: string;
}

class NavProfile extends React.Component<Props, State> {
  private wrapperRef: React.RefObject<HTMLDivElement>;

  public state: State = {
    menu: MenuOptions.Main,
    isMainMenuOpen: false,
    isHelpMenuOpen: false,
    isLocal: window.location.host.includes('localhost'),
    currentValue: '',
    globalTimezone: ''
  };

  constructor(props: Props) {
    super(props);
    this.wrapperRef = React.createRef();
    this.handleClickOutside = this.handleClickOutside.bind(this);
  }

  private goToMain = () => {
    this.setState({
      menu: MenuOptions.Main,
      transitionType: [TransitionType.TranslateLeft, TransitionType.TranslateRight]
    });
  };

  private goToTimezoneOptions = () => {
    this.setState({
      menu: MenuOptions.Timezone,
      transitionType: [TransitionType.TranslateRight, TransitionType.TranslateLeft]
    });
  };

  private openMainMenu = () => {
    this.setState(prevState => ({
      isMainMenuOpen: !prevState.isMainMenuOpen
    }));
  };

  private openHelpMenu = () => {
    this.setState(prevState => ({
      isHelpMenuOpen: !prevState.isHelpMenuOpen
    }));
  };

  private renderMenu() {
    if (this.state.menu === MenuOptions.Main) {
      return this.renderMainMenu();
    }
    return this.renderTimezoneOptionsMenu();
  }

  private logout = () => {
    Cookies.remove('expireDate');
    Cookies.remove('groups');
    Cookies.remove('profile');
    Cookies.remove('token');

    localStorage.removeItem('productOwner');
    localStorage.removeItem('productOwnerID');
  };

  private toggleTheme = () => {
    const nextValue = this.state.currentValue === 'tw-root--theme-dark' ? 'tw-root--theme-light' : 'tw-root--theme-dark';
    this.setCurrentTheme(nextValue);
    localStorage.setItem('currentThemeValue', nextValue);
  };

  private setCurrentTheme = (name: string) => {
    const wrapper = document.querySelector('html');

    if (wrapper) {
      wrapper.classList.remove('tw-root--theme-light');
      wrapper.classList.remove('tw-root--theme-dark');
      wrapper.classList.add(name);
    }

    this.setState({
      currentValue: name
    });
  };

  private renderMainMenu() {
    const { isLocal } = this.state;
    return (
      <Layout padding={1} key="main-menu">
        <Layout padding={{ x: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center}>
          <Avatar alt="Required alternative text." size={40} userLogin={Cookies.get('profile')} />
          <Layout display={Display.Flex} flexDirection={FlexDirection.Column} padding={{ x: 0.5 }}>
            <CoreText fontSize={FontSize.Size6} fontWeight={FontWeight.SemiBold}>
              {Cookies.get('profile')}
            </CoreText>
            <Layout display={Display.Flex} alignItems={AlignItems.Center} flexDirection={FlexDirection.Row}>
              <Layout padding={{ right: 0.5 }}>
                <Presence status={PresenceStatus.Online} />
              </Layout>
              <CoreText fontSize={FontSize.Size6}>Online</CoreText>
            </Layout>
          </Layout>
        </Layout>
        <DropDownMenuSeparator />
        <DropDownMenuItem label="Timezone" onClick={this.goToTimezoneOptions} figure={{ icon: SVGAsset.Global }} actionIcon={SVGAsset.AngleRight} />
        {isLocal && (
          <DropDownMenuItem
            label="cb/Anchor"
            figure={{ icon: SVGAsset.Github }}
            linkTo="https://git.xarth.tv/cb/anchor"
            targetBlank
            actionIcon={SVGAsset.Popout}
          />
        )}
        <DropDownMenuInputItem
          label="Dark Mode"
          onChange={this.toggleTheme}
          type={DropDownMenuInputItemType.Toggle}
          figure={{ icon: SVGAsset.Moon }}
          checked={this.state.currentValue === 'tw-root--theme-dark'}
        />
        {this.props.canMasquerade && <DropDownMenuItem label="Settings" figure={{ icon: SVGAsset.NavSettings }} linkTo="/settings" />}
        <DropDownMenuSeparator />
        <DropDownMenuItem label="Log Out" figure={{ icon: SVGAsset.NavLogout }} linkTo="/login" onClick={this.logout} />
      </Layout>
    );
  }

  private setGlobalTimezone = option => {
    localStorage.setItem('timezone', JSON.stringify(option));
    this.setState({ globalTimezone: option && option.value });
    document.location.reload();
  };

  private renderTimezoneOptionsMenu() {
    return (
      <Layout padding={1}>
        <DropDownMenuItem label="Back" figure={{ icon: SVGAsset.AngleLeft }} onClick={this.goToMain} />
        <SimpleBarReact style={{ maxHeight: 350 }}>
          {timezoneOptions.map(option => (
            <DropDownMenuItem key={option.value} onClick={() => this.setGlobalTimezone(option)} selected={option && option.value === this.state.globalTimezone}>
              <CoreText>{option.name}</CoreText>
              {option.offset === getDefaultTimezoneOption().offset && <Pill label="Local" />}
            </DropDownMenuItem>
          ))}
        </SimpleBarReact>
      </Layout>
    );
  }

  private renderHelpMenu() {
    return (
      <Layout overflow={Overflow.Hidden} padding={1}>
        <DropDownMenuItem linkTo="slack://channel?team=T0266V6GF&id=CTS04FHLP" targetBlank>
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Layout padding={{ right: 0.5 }} flexShrink={0}>
              <img src="/images/thumb-slack.png" alt="Slack logo" height="20" />
            </Layout>
            <Layout flexGrow={1}>Reach out on Slack</Layout>
          </Layout>
        </DropDownMenuItem>
        <DropDownMenuItem linkTo="https://jira.twitch.com/secure/CreateIssue!default.jspa?pid=27901" targetBlank>
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Layout padding={{ right: 0.5 }} flexShrink={0}>
              <img src="/images/thumb-jira.png" alt="Jira logo" height="20" />
            </Layout>
            <Layout flexGrow={1}>Submit a bug or feature request</Layout>
          </Layout>
        </DropDownMenuItem>
      </Layout>
    );
  }

  public componentDidMount() {
    if (localStorage.getItem('timezone') == null) {
      localStorage.setItem('timezone', JSON.stringify(getDefaultTimezoneOption()));
    }
    if (localStorage.getItem('timezone') != null) {
      this.setState({
        globalTimezone: JSON.parse(localStorage.getItem('timezone')).value
      });
    }
    document.addEventListener('mousedown', this.handleClickOutside);
    const previousTheme = localStorage.getItem('currentThemeValue');

    if (previousTheme) {
      this.setCurrentTheme(previousTheme);
      return;
    }

    this.setCurrentTheme('tw-root--theme-light');
  }

  public componentWillUnmount() {
    document.removeEventListener('mousedown', this.handleClickOutside);
  }

  private handleClickOutside(event) {
    if (this.wrapperRef && !this.wrapperRef.current.contains(event.target)) {
      this.setState({ isMainMenuOpen: false });
    }
  }

  render() {
    return (
      <Layout display={Display.Flex} fullWidth justifyContent={JustifyContent.End}>
        {this.props.isMasqueradeActive && (
          <Layout padding={{ x: 0.5 }}>
            <Tag label="Masquerade Active" linkTo="/settings" />
          </Layout>
        )}
        <Layout padding={{ x: 0.5 }}>
          <ClickOutDetector onClickOut={() => this.setState({ isHelpMenuOpen: false })}>
            <StyledLayout position={Position.Relative} fontSize={FontSize.Size6}>
              <Tooltip label="Help" direction={TooltipDirection.Bottom}>
                <ButtonIcon blurAfterClick icon={SVGAsset.QuestionMark} onClick={this.openHelpMenu} />
              </Tooltip>
              <Balloon direction={BalloonDirection.BottomRight} show={this.state.isHelpMenuOpen} size={BalloonSize.Small}>
                {this.renderHelpMenu()}
              </Balloon>
            </StyledLayout>
          </ClickOutDetector>
        </Layout>
        <Layout padding={{ x: 0.5 }}>
          <div ref={this.wrapperRef}>
            <StyledLayout position={Position.Relative} fontSize={FontSize.Size6}>
              <Tooltip label="Profile" direction={TooltipDirection.Bottom}>
                <ButtonIcon blurAfterClick icon={SVGAsset.NavProfile} onClick={this.openMainMenu} />
              </Tooltip>
              <Balloon direction={BalloonDirection.BottomRight} show={this.state.isMainMenuOpen} size={BalloonSize.Small}>
                <Layout overflow={Overflow.Hidden}>
                  <TransitionGroup duration={TransitionDuration.Medium} transitionType={this.state.transitionType}>
                    {this.renderMenu()}
                  </TransitionGroup>
                </Layout>
              </Balloon>
            </StyledLayout>
          </div>
        </Layout>
      </Layout>
    );
  }
}

export default NavProfile;
