import React, { useState } from 'react';
import {
  Layout,
  CoreText,
  TextType,
  FontWeight,
  Display,
  AlignItems,
  JustifyContent,
  StyledLayout,
  BorderRadius,
  Color,
  Position,
  Background,
  FontSize,
  Balloon,
  BalloonDirection,
  BalloonSize,
  FlexDirection,
  ZIndex
} from 'twitch-core-ui';
import { abbreviateNumber, isObjectEmpty } from '../utils/helpers';
import './styles/BarChart.scss';

function OverviewBarChart({ headerName = '', label = '', isRevenue = false, data = {} }) {
  if (isObjectEmpty(data)) {
    return null;
  } else {
    const yLabels = [];
    const numberOfYLabels = 7;

    const arrValues: Array<number> = Object.values(data);
    const barLabelMax = Math.max(...arrValues);

    const yLabelIterator = barLabelMax / numberOfYLabels;
    for (let i = numberOfYLabels; i > 0; i--) {
      yLabels.push(abbreviateNumber(i * yLabelIterator));
    }

    const months = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];

    return (
      <Layout display={Display.Block} margin={{ bottom: 5 }}>
        <Layout className="bar-chart--header" margin={{ bottom: 2 }}>
          <CoreText type={TextType.H4}>{headerName}</CoreText>
        </Layout>
        <Layout className="bar-chart--wrapper">
          <Layout display={Display.Flex}>
            <Layout className="y-labels" display={Display.Flex} alignItems={AlignItems.Center} flexDirection={FlexDirection.Column} margin={{ right: 0.5 }}>
              {yLabels.map((ylabel, i) => (
                <CoreText key={i} className="y-label" fontSize={FontSize.Size7} color={Color.Alt2}>
                  {ylabel}
                </CoreText>
              ))}
            </Layout>
            <StyledLayout className="bars" display={Display.Flex} alignItems={AlignItems.End} position={Position.Relative}>
              <Layout
                className="plotlines"
                position={Position.Absolute}
                zIndex={ZIndex.Below}
                fullWidth
                fullHeight
                attachRight
                attachTop
                display={Display.Flex}
                alignItems={AlignItems.Center}
                flexDirection={FlexDirection.Column}>
                {yLabels.map((_, i) => (
                  <Layout className="plotline" key={i} fullWidth />
                ))}
              </Layout>
              {Object.keys(data).map((item, i) => (
                <Bar key={i} height={(data[item] / barLabelMax) * 100} data={data} item={item} label={label} isRevenue={isRevenue} />
              ))}
            </StyledLayout>
          </Layout>
          <Layout
            className="x-labels"
            display={Display.Flex}
            alignItems={AlignItems.Center}
            justifyContent={JustifyContent.Center}
            margin={{ left: 5, top: 0.5 }}>
            {Object.keys(data).map((item, i) => {
              const monthName = months[parseInt(item.split('-')[1])];
              return (
                <CoreText key={i} className="x-label" fontSize={FontSize.Size7} color={Color.Alt2}>
                  {monthName && monthName.slice(0, 3)}
                </CoreText>
              );
            })}
          </Layout>
        </Layout>
      </Layout>
    );
  }
}

export default OverviewBarChart;

function Bar({ height, data, item, label, isRevenue }) {
  const [isBalloonOpen, setIsBalloonOpen] = useState(false);

  const months = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
  const dateArr = item.split('-');
  const tooltipHeader = `${months[parseInt(dateArr[1]) - 1]} ${dateArr[0]}`;

  return (
    <div className="bar--wrapper" style={{ height: `calc(${height}%)` }}>
      <div className="bar" style={{ height: `100%` }} onMouseEnter={() => setIsBalloonOpen(true)} onMouseLeave={() => setIsBalloonOpen(false)} />
      <Layout className="balloon--wrapper">
        <Balloon direction={BalloonDirection.Top} size={BalloonSize.Auto} show={isBalloonOpen} tail>
          <StyledLayout className="balloon__tooltip" background={Background.Overlay} borderRadius={BorderRadius.Medium} padding={0.5}>
            <CoreText type={TextType.H5} fontWeight={FontWeight.SemiBold}>
              {tooltipHeader}
            </CoreText>
            <Layout display={Display.Flex} alignItems={AlignItems.Center}>
              <StyledLayout className="balloon__label" background={Background.Accent} margin={{ right: 0.5 }} />
              <CoreText type={TextType.H6}>
                {isRevenue && '$'}
                {data[item].toLocaleString()} {label}
              </CoreText>
            </Layout>
          </StyledLayout>
        </Balloon>
      </Layout>
    </div>
  );
}
