import React, { useState } from 'react';
import {
  Layout,
  CoreText,
  TextType,
  FontWeight,
  Display,
  AlignItems,
  JustifyContent,
  FlexWrap,
  StyledLayout,
  BorderRadius,
  Color,
  Position,
  Background,
  FontSize,
  Interactable,
  InteractableType,
  Balloon,
  BalloonDirection,
  BalloonSize
} from 'twitch-core-ui';
import moment from 'moment';
import { getMomentDaysDiff, getMomentTimeAmPm, getMomentDate } from '../utils/dateTimeHelpers';
import './styles/Calendar.scss';

function OverviewCalendarEvents({ events }) {
  const tzConvertedEvents = {};
  for (let event in events) {
    const convertedDate = getMomentDate(events[event][0].start_time, 'YYYY-MM-DD');
    tzConvertedEvents[convertedDate] = events[event];
  }

  return (
    <Layout className="calendar--wrapper" display={Display.Flex} alignItems={AlignItems.Start} flexWrap={FlexWrap.Wrap}>
      <Calendar momentDate={moment()} events={tzConvertedEvents} />
      <Calendar momentDate={moment().add(1, 'months')} events={tzConvertedEvents} />
      <Calendar momentDate={moment().add(2, 'months')} events={tzConvertedEvents} />
      <Calendar momentDate={moment().add(3, 'months')} events={tzConvertedEvents} />
    </Layout>
  );
}

export default OverviewCalendarEvents;

function Calendar({ momentDate, events }) {
  const month = momentDate.month();
  const year = momentDate.year();
  const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];

  return (
    <Layout className="calendar" margin={{ right: 5, bottom: 5 }}>
      <Layout margin={{ bottom: 1 }}>
        <CoreText type={TextType.H4}>{momentDate.format('MMMM YYYY')}</CoreText>
      </Layout>
      <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center} margin={{ bottom: 0.5 }}>
        {dayNames.map(dayName => (
          <Layout className="calendar--dayNames" key={dayName} display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Center}>
            <CoreText type={TextType.H5} fontWeight={FontWeight.SemiBold}>
              {dayName}
            </CoreText>
          </Layout>
        ))}
      </Layout>
      <StyledLayout display={Display.Flex} alignItems={AlignItems.Center} flexWrap={FlexWrap.Wrap} borderRadius={BorderRadius.Medium}>
        {/* offset number of days */}
        {[...Array(parseInt(momentDate.startOf('month').format('e')))].map((_, i) => (
          <Layout className="calendar--day" key={month + i + 31}></Layout>
        ))}
        {/* start on first day of month */}
        {[...Array(momentDate.daysInMonth())].map((_, i) => {
          const day = i + 1;
          const activeCalendarDate = `${year}-${(month + 1).toString().padStart(2, '0')}-${day.toString().padStart(2, '0')}`;
          const activeEvents = events[activeCalendarDate];
          return <CalendarDay key={activeCalendarDate} day={day} activeCalendarDate={activeCalendarDate} activeEvents={activeEvents} />;
        })}
      </StyledLayout>
    </Layout>
  );
}

function CalendarDay({ day, activeCalendarDate, activeEvents }) {
  const [isBalloonOpen, setIsBalloonOpen] = useState(false);
  const isToday = activeCalendarDate === moment().format('YYYY-MM-DD');
  const isBeforeToday = activeCalendarDate < moment().format('YYYY-MM-DD');

  const renderEventData = (i, ev) => {
    const { event_name, start_time, end_time } = ev;
    const eventName = activeEvents > 1 ? `${i + 1}. ${event_name}` : event_name;
    const startTime = getMomentTimeAmPm(start_time);
    const endTime = getMomentTimeAmPm(end_time);
    const startEndDaysDiff = getMomentDaysDiff(start_time, end_time);

    return (
      <>
        <CoreText type={TextType.P}>{`${eventName}: ${startTime} - ${endTime}`}</CoreText>
        {startEndDaysDiff > 0 && (
          <CoreText fontSize={FontSize.Size8} color={Color.Live}>
            {`(+ ${startEndDaysDiff})`}
          </CoreText>
        )}
      </>
    );
  };

  if (activeEvents == null) {
    return (
      <Layout position={Position.Relative}>
        <Interactable type={InteractableType.Inverted} disabled={isBeforeToday}>
          <StyledLayout
            className="calendar--day"
            color={isToday ? Color.Overlay : Color.Base}
            key={activeCalendarDate}
            display={Display.Flex}
            alignItems={AlignItems.Center}
            justifyContent={JustifyContent.Center}
            position={Position.Relative}
            background={isToday ? Background.Accent : Background.Base}
            borderRight
            borderBottom>
            <Layout position={Position.Absolute} attachTop attachLeft margin={0.5}>
              <CoreText fontSize={FontSize.Size7}>{day}</CoreText>
            </Layout>
          </StyledLayout>
        </Interactable>
      </Layout>
    );
  } else {
    return (
      <Layout position={Position.Relative}>
        <Interactable type={InteractableType.Inverted} onMouseEnter={() => setIsBalloonOpen(true)} onMouseLeave={() => setIsBalloonOpen(false)}>
          <StyledLayout
            className="calendar--day"
            color={isToday ? Color.Overlay : Color.Base}
            key={activeCalendarDate}
            display={Display.Flex}
            alignItems={AlignItems.Center}
            justifyContent={JustifyContent.Center}
            position={Position.Relative}
            background={isToday ? Background.Accent : activeEvents.length === 2 ? Background.Alt2 : Background.Alt}
            borderRight
            borderBottom>
            <Layout position={Position.Absolute} attachTop attachLeft margin={0.5}>
              <CoreText fontSize={FontSize.Size7}>{day}</CoreText>
            </Layout>
            <CoreText type={TextType.H5} fontWeight={FontWeight.Bold}>
              {activeEvents.length}
            </CoreText>
          </StyledLayout>
        </Interactable>
        <Layout className="balloon--wrapper">
          <Balloon direction={BalloonDirection.Top} size={BalloonSize.Auto} show={isBalloonOpen} tail>
            <StyledLayout className="balloon__tooltip" background={Background.Overlay} borderRadius={BorderRadius.Medium} padding={0.5}>
              <CoreText type={TextType.H5} fontWeight={FontWeight.SemiBold}>
                {moment(activeCalendarDate).format('dddd, MMMM D, YYYY')}
              </CoreText>
              {activeEvents.map((ev, i) => {
                return (
                  <Layout display={Display.Flex} key={`${activeCalendarDate}-${i}`}>
                    {renderEventData(i, ev)}
                  </Layout>
                );
              })}
            </StyledLayout>
          </Balloon>
        </Layout>
      </Layout>
    );
  }
}
