import React, { useState } from 'react';
import { Layout, CoreText, TextType, Display, AlignItems, JustifyContent, Button, ButtonType, SVG, SVGAsset, FlexWrap } from 'twitch-core-ui';
import { useLocalStorage } from '@rehooks/local-storage';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import ProductsList from './products-list';
import ProductsPlaceholder from './products-placeholder';
import OverviewCalendarEvents from './overview-calendar-events';
import OverviewBarChart from './overview-bar-chart';
import OverviewBarChartPlaceholder from './overview-bar-chart-placeholder';
import MetricsConsole from './metrics-console';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST, useFetchGET } from '../hooks/useFetch';
import useUpdateEffect from '../hooks/useUpdateEffect';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function Overview() {
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const [filters, setFilters] = useState([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);

  // calendar
  /* eslint-disable @typescript-eslint/no-unused-vars */
  let [calendarData, calendarLoading, calendarError, fetchCalendarData] = useFetchGET(
    `${ATLAS_BASE_URL}/event/get-active-calendar-events/${selectedContentCreator}`
  );
  const calendarEvents = calendarData == null || calendarData.data == null ? [] : calendarData.data.events_by_day;
  const metrics = calendarData == null || calendarData.metrics == null ? [] : calendarData.metrics;

  // analytics
  /* eslint-disable @typescript-eslint/no-unused-vars */
  let [analyticsData, analyticsLoading, analyticsError, fetchAnalyticsData] = useFetchPOST(
    `${ATLAS_BASE_URL}/event/get-aggregate-pcc-stats`,
    {
      pcc_ids: [selectedContentCreator]
    }
  );
  const analytics = analyticsData == null || analyticsData.data == null ? [] : analyticsData.data;
  const hoursWatched = analyticsLoading ? {} : createDataObjFromAnalytics(analytics, 'hours_watched');
  const totalRevenue = analyticsLoading ? {} : createDataObjFromAnalytics(analytics, 'total_revenue');

  // products
  let [productsData, productsLoading, productsError, fetchProductsData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-product-list`, {
    limit: 10,
    page: 0,
    eventtype: 1,
    'req-cols': ['*'],
    filters: filters,
    sortdata: [{ ord: 0, col: 'product_name', asc: true }]
  });
  const productsList = productsData == null || productsData.data == null ? [] : productsData.data.items;

  useUpdateEffect(() => {
    setFilters([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);
  }, [selectedContentCreator]);

  useUpdateEffect(() => {
    fetchProductsData();
  }, [filters]);

  useUpdateEffect(() => {
    fetchCalendarData();
    fetchAnalyticsData();
  }, [selectedContentCreator]);

  function createDataObjFromAnalytics(data, type = '') {
    if (data == null) return;
    const dataObj = {};
    const months = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
    const today = new Date();

    for (let i = 0; i < months.length; i++) {
      let isPrevYear = true;
      let monthIndex = today.getMonth() + 2 + i;
      if (monthIndex > 12) {
        isPrevYear = false;
      }
      let dataFormattedDate = `${isPrevYear ? today.getFullYear() - 1 : today.getFullYear()}-${(monthIndex === 12 ? monthIndex : monthIndex % 12)
        .toString()
        .padStart(2, '0')}-01T00:00:00`;
      dataObj[dataFormattedDate] = 0;
    }

    data.forEach(item => {
      if (dataObj[item.measurement_date] !== undefined) {
        dataObj[item.measurement_date] = item[type];
      }
    });

    return dataObj;
  }

  return (
    <>
      <Layout>
        <Layout className="overview--section" margin={{ bottom: 5 }}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between} margin={{ bottom: 1 }}>
            <Layout display={Display.Flex} alignItems={AlignItems.Center}>
              <SVG asset={SVGAsset.Events} />
              <Layout margin={{ left: 1 }}>
                <CoreText type={TextType.H3}>Events</CoreText>
              </Layout>
            </Layout>
            <Layout display={Display.Flex}>
              <Layout margin={{ x: 1 }}>
                <Button type={ButtonType.Secondary} linkTo="/events">
                  Manage Events
                </Button>
              </Layout>
              <Button linkTo="/events/create">Create New Event</Button>
            </Layout>
          </Layout>
          {calendarError && <ErrorAlert error={calendarError} />}
          <OverviewCalendarEvents events={calendarEvents} />
        </Layout>

        <Layout className="overview--section">
          <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between} margin={{ bottom: 1 }}>
            <Layout display={Display.Flex} alignItems={AlignItems.Center}>
              <SVG asset={SVGAsset.NavDashboard} />
              <Layout margin={{ left: 1 }}>
                <CoreText type={TextType.H3}>Analytics</CoreText>
              </Layout>
            </Layout>
          </Layout>
          {analyticsError && <ErrorAlert error={analyticsError} />}
          {analyticsLoading ? (
            <OverviewBarChartPlaceholder />
          ) : analytics && analytics.length === 0 ? (
            <Layout margin={{ bottom: 5 }}>
              <NoResults results="Analytics" />
            </Layout>
          ) : (
            <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between} flexWrap={FlexWrap.Wrap}>
              <OverviewBarChart headerName="Aggregate Hours Watched" label="hours watched" data={hoursWatched} />
              <OverviewBarChart headerName="Total Revenue" isRevenue data={totalRevenue} />
            </Layout>
          )}
        </Layout>

        <Layout className="overview--section" margin={{ bottom: 5 }}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between} margin={{ bottom: 1 }}>
            <Layout display={Display.Flex} alignItems={AlignItems.Center}>
              <SVG asset={SVGAsset.Trophy} />
              <Layout margin={{ left: 1 }}>
                <CoreText type={TextType.H3}>Products</CoreText>
              </Layout>
            </Layout>
            <Layout display={Display.Flex}>
              <Layout margin={{ x: 1 }}>
                <Button type={ButtonType.Secondary} linkTo="/products">
                  Manage Products
                </Button>
              </Layout>
              <Button linkTo="/products/create">Create New Product</Button>
            </Layout>
          </Layout>
          {productsError && <ErrorAlert error={productsError} />}
          {productsLoading ? (
            <ProductsPlaceholder />
          ) : productsList && productsList.length === 0 ? (
            <NoResults results="Products" />
          ) : (
            <Layout className="products--card-wrapper" display={Display.Flex} flexWrap={FlexWrap.Wrap}>
              <ProductsList products={productsList} />
            </Layout>
          )}
        </Layout>
      </Layout>
      <MetricsConsole metrics={metrics} loading={calendarLoading} name="Overview Calendar" />
    </>
  );
}

export default Overview;
