import React, { useState } from 'react';
import { useHistory } from 'react-router-dom';
import Cookies from 'js-cookie';
import {
  Layout,
  CoreText,
  TextType,
  StyledLayout,
  FormGroup,
  FormGroupOrientation,
  Input,
  InputType,
  Select,
  Button,
  ButtonType,
  Display,
  JustifyContent,
  ButtonState,
  Background,
  BorderRadius
} from 'twitch-core-ui';
import FilePicker from './file-picker';
import Loading from './loading';
import ErrorAlert from './error-alert';
import { ATLAS_BASE_URL } from '../utils/config';
import { fetchWithTimeout } from '../utils/helpers';
import './styles/Form.scss';
import { useDataContext } from '../context/data-context';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function ProductsCreate() {
  const history = useHistory();
  const [isFormSubmitting, setIsFormSubmitting] = useState(false);
  const [error, setError] = useState('');

  // format options
  const { allTopicsFormatsOptions, loading } = useDataContext();

  // product form data
  // hidden
  const [productId] = useState('0');
  // visible
  const [productName, setProductName] = useState('');
  const [description, setDescription] = useState('');
  const [topic, setTopic] = useState('');
  const [frontPagePromotionBanner, setFrontPagePromotionBanner] = useState(null);

  const [bannerErrorMessage, setBannerErrorMessage] = useState('');

  const pccID = localStorage.getItem(PCCID_LOCAL_STORAGE_KEY);

  const handleTopicSelect = topic => {
    if (topic) {
      setTopic(topic);
    }
  };

  const submitForm = () => {
    let product = {
      product_id: productId,
      product_name: productName,
      product_description: description,
      topic: topic,
      front_page_banner: frontPagePromotionBanner,
      is_active: true,
      premium_content_creator_id: pccID,
      // The following fields are deprecated, but Resonance will reject the request if they are null:
      sub_topic: '',
      contract_id: ''
    };

    const body = Object.keys(product)
      .filter(e => product[e] !== null)
      .reduce((o, e) => {
        o[e] = product[e];
        return o;
      }, {});

    setIsFormSubmitting(true);
    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(body)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-product`, options);
        const json = await response.json();
        console.log(`/edit-product:`, json);

        if (json.success) {
          history.push('/products');
        } else {
          setError(!!json.error_message ? json.error_message : 'Edit Product request not successful');
          setIsFormSubmitting(false);
        }
      } catch (err) {
        console.error(`/edit-product:`, err);
        setError(err);
        setIsFormSubmitting(false);
      }
    }
    fetchData();
  };

  const isFormValid = !!productName && !!topic && bannerErrorMessage.length === 0;

  return (
    <Layout className="form--wrapper" margin="auto">
      <Layout padding={{ bottom: 1 }}>
        <CoreText type={TextType.H3}>Create New Product</CoreText>
      </Layout>
      <Layout margin={{ bottom: 2 }}>
        <CoreText type={TextType.H4}>Product Details</CoreText>
      </Layout>
      {loading ? (
        <Loading />
      ) : (
        <>
          <StyledLayout background={Background.Base} borderRadius={BorderRadius.Medium} border>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Name" hint="Name your product" required>
                <Input type={InputType.Text} placeholder="Product Name" value={productName} onChange={e => setProductName(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Description" hint="Describe this product">
                <Input type={InputType.Text} placeholder="Product Description" value={description} onChange={e => setDescription(e.currentTarget.value)} />
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup orientation={FormGroupOrientation.Horizontal} label="Topic" hint="The product's topic or category" required>
                <Select value={topic} onChange={e => handleTopicSelect(e.currentTarget.value)}>
                  <option value="">Select Topic</option>
                  {allTopicsFormatsOptions?.map(option => (
                      <option value={option.topic} key={`topic-${option.topic}`}>
                        {option.topic}
                      </option>
                    ))}
                </Select>
              </FormGroup>
            </StyledLayout>
            <StyledLayout padding={2} borderBottom>
              <FormGroup
                orientation={FormGroupOrientation.Horizontal}
                label="Product Image"
                hint="Dimensions: 280x156; Max File Size: 100KB; File Format: JPG or PNG"
                error={!!bannerErrorMessage}
                errorMessage={bannerErrorMessage}>
                <FilePicker
                  allowedFileTypes={['.jpg', '.png']}
                  file={frontPagePromotionBanner}
                  setFile={setFrontPagePromotionBanner}
                  errorMessage={bannerErrorMessage}
                  setErrorMessage={setBannerErrorMessage}
                />
              </FormGroup>
            </StyledLayout>
          </StyledLayout>

          {error && (
            <Layout margin={{ y: 3 }}>
              <ErrorAlert error={error} />
            </Layout>
          )}

          <Layout margin={{ y: 3 }} display={Display.Flex} justifyContent={JustifyContent.End}>
            <Layout margin={{ right: 1 }}>
              <Button
                type={ButtonType.Secondary}
                linkTo="/products"
                state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default}
                disabled={isFormSubmitting}>
                Cancel
              </Button>
            </Layout>
            <Button onClick={submitForm} state={isFormSubmitting ? ButtonState.Loading : ButtonState.Default} disabled={isFormSubmitting || !isFormValid}>
              Submit
            </Button>
          </Layout>
        </>
      )}
    </Layout>
  );
}

export default ProductsCreate;
