import React from 'react';
import Cookies from 'js-cookie';
import {
  CoreText,
  TextType,
  Layout,
  Display,
  SVG,
  SVGAsset,
  Placeholder,
  AlignItems,
  Tag,
  FlexWrap,
  HoverAccentEffect,
  Position,
  JustifyContent,
  Interactable,
  InteractableType,
  AlignSelf,
  Tooltip,
  TooltipDirection
} from 'twitch-core-ui';
import MoreActions from './more-actions';
import { ATLAS_BASE_URL } from '../utils/config';
import { useSelectedPermissions } from '../context/permissions-context';
import { fetchWithTimeout } from '../utils/helpers';

function ProductsList({ products, editProduct = undefined }) {
  return (
    <Layout className="products--card-wrapper" display={Display.Flex} flexWrap={FlexWrap.Wrap}>
      {products && products.map(product => <ProductCard key={product.product_id} product={product} editProduct={editProduct} />)}
    </Layout>
  );
}

function ProductCard({ product, editProduct }) {
  const permissions = useSelectedPermissions();
  const canEdit = permissions.includes('Can Manage Products') || permissions.includes('Manage Owned Channels');

  const deleteProduct = () => {
    let obj = {
      ...product,
      is_active: false
    };

    const body = Object.keys(obj)
      .filter(e => obj[e] !== null)
      .reduce((o, e) => {
        o[e] = obj[e];
        return o;
      }, {});

    async function fetchData() {
      let token = Cookies.get('token');
      const options = {
        method: 'POST',
        headers: {
          Accept: 'application/json',
          'Content-Type': 'application/json',
          Authorization: `Bearer ${token}`
        },
        body: JSON.stringify(body)
      };
      try {
        const response = await fetchWithTimeout(`${ATLAS_BASE_URL}/event/edit-product`, options);
        const json = await response.json();
        console.log(`/edit-product:`, json);
        document.location.reload();
      } catch (error) {
        console.error(`/edit-product:`, error);
        // TODO: on error display error message somewhere
      }
    }
    fetchData();
  };

  const { product_name, product_description, topic, sub_topic, count_of_channels, count_of_upcoming_events, front_page_asset } = product;

  return (
    <Layout className="products--card" margin={{ bottom: 5, right: 4 }}>
      <div style={{ width: 280 }}>
        <Layout className="products--btn-wrapper" margin={{ bottom: 1 }} position={Position.Relative}>
          {editProduct ? (
            <HoverAccentEffect>
              <Interactable
                type={InteractableType.Alpha}
                onClick={() => {
                  if (canEdit) {
                    editProduct(product);
                  }
                }}>
                {front_page_asset ? <img src={`${front_page_asset}`} alt="Front Page Banner" /> : <Placeholder width={280} height={156} />}
              </Interactable>
            </HoverAccentEffect>
          ) : front_page_asset ? (
            <img src={`${front_page_asset}`} alt="Front Page Banner" />
          ) : (
            <Placeholder width={280} height={156} />
          )}
        </Layout>
        <Layout margin={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center} justifyContent={JustifyContent.Between}>
          <CoreText type={TextType.H4} bold ellipsis>
            {product_name}
          </CoreText>
          <MoreActions editAction={editProduct ? () => editProduct(product) : undefined} deleteAction={deleteProduct} />
        </Layout>
        <Layout margin={{ bottom: 0.5 }} display={Display.Flex} alignItems={AlignItems.Center}>
          <Layout display={Display.Flex} alignItems={AlignItems.Center} margin={{ right: 2 }}>
            <Tooltip label={`${count_of_channels} channels`} direction={TooltipDirection.Right}>
              <SVG asset={SVGAsset.Webcam} />
              <Layout margin={{ left: 0.5 }}>
                <CoreText type={TextType.H5} bold>
                  {count_of_channels}
                </CoreText>
              </Layout>
            </Tooltip>
          </Layout>
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Tooltip label={`${count_of_upcoming_events} events`} direction={TooltipDirection.Right}>
              <SVG asset={SVGAsset.Events} />
              <Layout margin={{ left: 0.5 }}>
                <CoreText type={TextType.H5} bold>
                  {count_of_upcoming_events}
                </CoreText>
              </Layout>
            </Tooltip>
          </Layout>
        </Layout>
        <Layout margin={{ bottom: 0.5 }} display={Display.Flex} flexWrap={FlexWrap.Wrap}>
          <Layout margin={{ right: 0.5 }} display={Display.Flex} alignSelf={AlignSelf.Start}>
            <Tag label={topic || 'No Topic'} onClick={e => e.preventDefault()} />
          </Layout>
          <Layout display={Display.Flex} alignSelf={AlignSelf.Start}>
            <Tag label={sub_topic || 'No Sub-Topic'} onClick={e => e.preventDefault()} />
          </Layout>
        </Layout>
        <CoreText type={TextType.P}>{product_description || 'No product description'}</CoreText>
      </div>
    </Layout>
  );
}

export default ProductsList;
