import React, { useState } from 'react';
import { useHistory } from 'react-router-dom';
import { Layout } from 'twitch-core-ui';
import ErrorAlert from './error-alert';
import NoResults from './no-results';
import PaginationWrapper from './pagination-wrapper';
import ProductsHeader from './products-header';
import ProductsList from './products-list';
import ProductsPlaceholder from './products-placeholder';
import MetricsConsole from './metrics-console';
import { ATLAS_BASE_URL } from '../utils/config';
import { useFetchPOST } from '../hooks/useFetch';
import { useLocalStorage } from '@rehooks/local-storage';
import useUpdateEffect from '../hooks/useUpdateEffect';
import './styles/Products.scss';
import { PCCID_LOCAL_STORAGE_KEY } from '../utils/constants';

function Products() {
  const history = useHistory();
  const [selectedContentCreator] = useLocalStorage(PCCID_LOCAL_STORAGE_KEY);
  const searchParams = new URLSearchParams(history.location.search);
  const [currentPage, setCurrentPage] = useState(parseInt(searchParams.get('page')) || 1);
  const [filters, setFilters] = useState([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);

  let options = {
    limit: 100,
    page: currentPage - 1,
    eventtype: 1,
    'req-cols': ['*'],
    filters: filters,
    sortdata: [{ ord: 0, col: 'product_name', asc: true }]
  };
  let [data, loading, error, fetchData] = useFetchPOST(`${ATLAS_BASE_URL}/event/get-product-list`, options);
  const products = data == null || data.data == null ? [] : data.data.items;
  const totalPages = data == null || data.data == null ? 0 : data.data.total_pages;
  const metrics = data == null || data.metrics == null ? [] : data.metrics;

  useUpdateEffect(() => {
    setFilters([{ 'filter-type': 1, key: 'premium_content_creator_id', v1: selectedContentCreator }]);
  }, [selectedContentCreator]);

  useUpdateEffect(() => {
    setCurrentPage(currentPage);
    history.push({
      search: currentPage > 1 ? `?page=${currentPage}` : ''
    });
    fetchData(); /* eslint-disable react-hooks/exhaustive-deps */
  }, [currentPage, history, filters]);

  const editProduct = product => {
    history.push(`/products/edit/${product.product_id}`);
  };

  return (
    <>
      <Layout className="products">
        <ProductsHeader />
        {error && <ErrorAlert error={error} />}
        {loading ? (
          <ProductsPlaceholder />
        ) : products && products.length === 0 ? (
          <NoResults results="Products" />
        ) : (
          <>
            <ProductsList products={products} editProduct={editProduct} />
            {totalPages > 1 && <PaginationWrapper currentPage={currentPage} totalPages={totalPages} setCurrentPage={setCurrentPage} />}
          </>
        )}
      </Layout>
      <MetricsConsole metrics={metrics} loading={loading} name="Products" />
    </>
  );
}

export default Products;
